<?php
require 'config.php'; // ✅ $pdo already exists here

// ===== LOG SETUP =====
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/pool.log'); // Log to pool.log file

// ✅ Safety check in case DB fails
if (!$pdo) {
    error_log("Database connection failed");
    echo json_encode(['ok' => false, 'message' => 'Database connection failed']);
    exit;
}

// Input parsing
$action = $_GET['action'] ?? ($_POST['action'] ?? '');
$input = json_decode(file_get_contents("php://input"), true) ?? $_POST ?? [];
$user_id = intval($input['user_id'] ?? ($_GET['user_id'] ?? 0));

error_log("Request received: Action=$action, UserID=$user_id, Input=" . json_encode($input));

if ($user_id <= 0) {
    error_log("Invalid user: UserID=$user_id");
    echo json_encode(['ok' => false, 'message' => 'Invalid user']);
    exit;
}

// ✅ Task Definitions
$tasks = [
    'daily_bonus' => [
        'type' => 'daily',
        'description' => 'Claim Daily Mega Reward',
        'rewards' => ['balance' => 100, 'keys' => 0],
        'reset_interval' => 86400
    ],
    'daily_draws' => [
        'type' => 'daily',
        'description' => 'Do 50 draws to get keys',
        'rewards' => ['balance' => 0, 'keys' => 5],
        'reset_interval' => 86400
    ],
    'daily_share' => [
        'type' => 'daily',
        'description' => 'Share Daily to Earn',
        'rewards' => ['balance' => 50, 'keys' => 1],
        'reset_interval' => 86400
    ],
    'limited_subscribe' => [
        'type' => 'limited',
        'description' => 'Subscribe to our Telegram Channel',
        'rewards' => ['balance' => 150, 'keys' => 2],
        'limit' => 1
    ],
    'limited_invite_100' => [
        'type' => 'limited',
        'description' => 'Invite 100 Users',
        'rewards' => ['balance' => 1000, 'keys' => 10],
        'limit' => 1
    ]
];

// ✅ Ensure user exists
function ensureUserExists($pdo, $user_id) {
    try {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        if (!$stmt->fetch()) {
            $insert = $pdo->prepare("INSERT INTO users (id, balance, `keys`) VALUES (?, 0, 0)");
            $insert->execute([$user_id]);
            error_log("Created new user: $user_id");
        }
    } catch (Exception $e) {
        error_log("Error ensuring user exists: " . $e->getMessage());
    }
}

// ✅ Get user stats
if ($action === 'get_user_stats') {
    ensureUserExists($pdo, $user_id);

    try {
        $stmt = $pdo->prepare("SELECT balance, `keys` FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $data = $stmt->fetch(PDO::FETCH_ASSOC) ?? ['balance' => 0, 'keys' => 0];
        echo json_encode(['ok' => true, 'data' => $data]);
    } catch (Exception $e) {
        error_log("Error in get_user_stats: " . $e->getMessage());
        echo json_encode(['ok' => false, 'message' => 'Failed to get user stats']);
    }
    exit;
}

// ✅ Helper - Find Available Tasks
function getAvailableTasks($pdo, $tasks, $user_id) {
    $available = [];
    try {
        $stmt = $pdo->prepare("SELECT task_key, completed_at, count FROM user_tasks WHERE user_id = ?");
        $stmt->execute([$user_id]);
        $userTasks = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $userTaskMap = [];
        foreach ($userTasks as $t) {
            $userTaskMap[$t['task_key']] = $t;
        }

        foreach ($tasks as $key => $task) {
            $canDo = true;

            if (isset($userTaskMap[$key])) {
                $ut = $userTaskMap[$key];

                if ($task['type'] === 'daily') {
                    $last = strtotime($ut['completed_at']);
                    $canDo = (time() - $last) >= $task['reset_interval'];
                } elseif ($task['type'] === 'limited') {
                    $canDo = $ut['count'] < $task['limit'];
                }
            }

            if ($canDo) $available[$key] = $task;
        }
    } catch (Exception $e) {
        error_log("Error in getAvailableTasks: " . $e->getMessage());
    }
    return $available;
}

// ✅ Fetch available tasks
if ($action === 'get_tasks') {
    ensureUserExists($pdo, $user_id);
    $available = getAvailableTasks($pdo, $tasks, $user_id);
    echo json_encode(['ok' => true, 'tasks' => $available]);
    exit;
}

// ✅ Complete task
if ($action === 'complete_task') {
    ensureUserExists($pdo, $user_id);
    $task_key = $input['task_key'] ?? '';

    if (!isset($tasks[$task_key])) {
        error_log("Invalid task: $task_key for user $user_id");
        echo json_encode(['ok' => false, 'message' => 'Invalid task']);
        exit;
    }

    $available = getAvailableTasks($pdo, $tasks, $user_id);
    if (!isset($available[$task_key])) {
        error_log("Task already done: $task_key for user $user_id");
        echo json_encode(['ok' => false, 'message' => 'Task already done']);
        exit;
    }

    try {
        $check = $pdo->prepare("SELECT * FROM user_tasks WHERE user_id = ? AND task_key = ?");
        $check->execute([$user_id, $task_key]);
        $row = $check->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $pdo->prepare("UPDATE user_tasks SET completed_at = NOW(), count = count + 1 WHERE id = ?")
                ->execute([$row['id']]);
        } else {
            $pdo->prepare("INSERT INTO user_tasks (user_id, task_key, completed_at, count)
                           VALUES (?, ?, NOW(), 1)")
                ->execute([$user_id, $task_key]);
        }

        $reward = $tasks[$task_key]['rewards'];
        $pdo->prepare("UPDATE users SET balance = balance + ?, `keys` = `keys` + ? WHERE id = ?")
            ->execute([$reward['balance'], $reward['keys'], $user_id]);

        error_log("Task completed: $task_key for user $user_id");
        echo json_encode(['ok' => true, 'message' => 'Task completed', 'rewards' => $reward]);
    } catch (Exception $e) {
        error_log("Error completing task $task_key for user $user_id: " . $e->getMessage());
        echo json_encode(['ok' => false, 'message' => 'Failed to complete task']);
    }
    exit;
}

error_log("Invalid action: $action for user $user_id");
echo json_encode(['ok' => false, 'message' => 'Invalid action']);
exit;