<?php
// Include dependencies
require 'config.php';

header('Content-Type: application/json');

// Ensure PDO is defined
if (!isset($pdo) || !($pdo instanceof PDO)) {
    echo json_encode(['ok' => false, 'message' => 'Database connection not initialized.']);
    exit;
}

// Get user_id from URL parameter
$user_id = isset($_GET['user_id']) ? $_GET['user_id'] : null;
if (empty($user_id)) {
    echo json_encode(['ok' => false, 'message' => 'Missing user_id parameter.']);
    exit;
}

// Fetch user and key_buy_address
try {
    $stmt = $pdo->prepare("SELECT id, user_id, key_buy_address FROM users WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        echo json_encode(['ok' => false, 'message' => 'User not found']);
        exit;
    }
} catch (PDOException $e) {
    echo json_encode(['ok' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    exit;
}

// Return existing address if available
if (!empty($user['key_buy_address'])) {
    echo json_encode([
        'ok' => true,
        'message' => 'Existing payment address retrieved successfully',
        'address' => $user['key_buy_address']
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}

// Prepare OxaPay API request
$oxapay_url = 'https://api.oxapay.com/v1/payment/static-address';
$payload = [
    'network' => 'USDT',
    'to_currency' => 'USDT',
    'callback_url' => "https://wenepay.cyou/scratch/api/key-wehook.php?user_id=$user_id"
];
$headers = [
    'Content-Type: application/json',
    'merchant_api_key: ' . ($config['oxapay_api_key'] ?? '')
];

// Initialize cURL
$ch = curl_init($oxapay_url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

// Execute cURL request
$response = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

// Decode OxaPay response
$response_data = json_decode($response, true);
if (json_last_error() !== JSON_ERROR_NONE || !is_array($response_data)) {
    echo json_encode(['ok' => false, 'message' => 'Invalid response from OxaPay API']);
    exit;
}

// Check for OxaPay error
if (isset($response_data['error']) && !empty($response_data['error'])) {
    echo json_encode([
        'ok' => false,
        'message' => 'OxaPay API error: ' . ($response_data['error']['message'] ?? 'Unknown error')
    ]);
    exit;
}

// Validate response data
if (!isset($response_data['data']['address'])) {
    echo json_encode(['ok' => false, 'message' => 'Missing address in OxaPay response']);
    exit;
}

// Save address to users.key_buy_address
try {
    $pdo->beginTransaction();
    $stmt = $pdo->prepare("UPDATE users SET key_buy_address = ? WHERE user_id = ?");
    $stmt->execute([$response_data['data']['address'], $user_id]);
    $pdo->commit();
} catch (PDOException $e) {
    $pdo->rollBack();
    echo json_encode(['ok' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    exit;
}

// Final response
echo json_encode([
    'ok' => true,
    'message' => 'Payment address generated and saved successfully',
    'address' => $response_data['data']['address']
], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
?>