<?php
require 'config.php';

header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('Asia/Kolkata');
ini_set('display_errors', 0);

function json_ok($payload = []) {
    return json_encode(array_merge(['ok' => true], $payload), JSON_UNESCAPED_UNICODE);
}
function json_fail($message, $code = 400) {
    http_response_code($code);
    return json_encode(['ok' => false, 'message' => $message], JSON_UNESCAPED_UNICODE);
}

/* read input */
$body = file_get_contents('php://input');
$input = json_decode($body, true);
if (!$input || empty($input['invoice_id']) || empty($input['coin']) || empty($input['user_id'])) {
    echo json_fail('Invalid request. Expecting invoice_id, coin and user_id.', 400);
    exit;
}
$track_id = trim($input['invoice_id']);
$coin = strtolower(trim($input['coin']));
$user_id = trim($input['user_id']);

/* validate coin */
$allowed_coins = ['usdt', 'trx', 'doge'];
if (!in_array($coin, $allowed_coins)) {
    echo json_fail('Unsupported coin selected.', 400);
    exit;
}

/* merchant key */
$merchant_api_key = "JSCWPO-YHHJEV-LNCH6I-USRJEC";

/* call OxaPay invoice status API */
try {
    $url = "https://api.oxapay.com/v1/payment/" . urlencode($track_id);

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "merchant_api_key: {$merchant_api_key}",
        "accept: application/json",
        "Content-Type: application/json"
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);

    $resp = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if ($resp === false) {
        $err = curl_error($ch);
        curl_close($ch);
        echo json_fail("Curl error: " . $err, 500);
        exit;
    }
    curl_close($ch);

    $json = json_decode($resp, true);
    if (!is_array($json) || !isset($json['status']) || $json['status'] != 200) {
        echo json_fail('Failed to fetch invoice status. HTTP Code: ' . $http_code . '. Response: ' . $resp, 500);
        exit;
    }

    $data = $json['data'] ?? [];
    $invoice_status = $data['status'] ?? 'unknown';
    $usd_amount = (float)($data['amount'] ?? 0);

    if ($invoice_status === 'paid') {
        if (empty($data['txs'])) {
            echo json_fail("No transaction found for paid invoice.", 500);
            exit;
        }
        $currency = strtoupper($data['txs'][0]['currency'] ?? '');
        $crypto_amount = (float)($data['txs'][0]['amount'] ?? 0);

        // Check that invoice currency matches selected coin (for safety)
        $map = [
            'usdt' => 'USDT',
            'trx' => 'TRX',
            'doge' => 'DOGE'
        ];
        if ($currency !== ($map[$coin] ?? '')) {
            echo json_fail("Invoice coin mismatch. Expected {$map[$coin]} got {$currency}.", 400);
            exit;
        }

        // Validate user existence
        try {
            $stmt = $pdo->prepare("SELECT id FROM users WHERE user_id = ?");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$user) {
                echo json_fail('User not found.', 400);
                exit;
            }
            $user_internal_id = $user['id'];
        } catch (PDOException $e) {
            error_log("User validation failed: " . $e->getMessage(), 3, 'error.log');
            echo json_fail('Database error validating user: ' . $e->getMessage(), 500);
            exit;
        }

        // Update task_balance with USD amount
        try {
            $stmt = $pdo->prepare("SELECT task_balance FROM mytask WHERE id = ?");
            $stmt->execute([$user_internal_id]);
            $mytask = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$mytask) {
                $stmt = $pdo->prepare("INSERT INTO mytask (id, tasks, task_balance, deposit_address) VALUES (?, ?, ?, ?)");
                $stmt->execute([$user_internal_id, '[]', 0.00, null]);
                $mytask = ['task_balance' => 0.00];
            }

            $new_task_balance = $mytask['task_balance'] + $usd_amount;
            $stmt = $pdo->prepare("UPDATE mytask SET task_balance = ? WHERE id = ?");
            $stmt->execute([$new_task_balance, $user_internal_id]);
        } catch (PDOException $e) {
            error_log("Task balance update failed: " . $e->getMessage(), 3, 'error.log');
            echo json_fail('Database error updating task balance: ' . $e->getMessage(), 500);
            exit;
        }

        // Optional: Send Telegram notification with crypto amount
        $telegram_message = urlencode("Your transaction (Track ID: {$track_id}) of {$crypto_amount} {$currency} has been paid successfully.");
        $telegram_url = "https://api.telegram.org/bot{$config['bot_token']}/sendMessage?chat_id={$user_id}&text={$telegram_message}";

        $ch = curl_init($telegram_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);
        curl_close($ch);

        echo json_ok([
            'message' => 'Deposit confirmed ✅',
            'amount'  => $usd_amount,
            'coin'    => $currency,
            'status'  => $invoice_status
        ]);
    } else {
        echo json_fail("Invoice status: {$invoice_status}", 200);
    }
} catch (Exception $e) {
    echo json_fail('Invoice verification failed: ' . $e->getMessage(), 500);
    exit;
}
?>