<?php
// registration.php
require 'config.php';
require 'init_data_check.php';

// Secure headers
header('Content-Type: application/json');
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header("Content-Security-Policy: default-src 'none'");
header('Access-Control-Allow-Origin: *');

// ---- Input ----
$data = json_decode(file_get_contents("php://input"), true);
$user_id   = (int)($data['user_id'] ?? 0);
$name      = filter_var(trim($data['name'] ?? ''), FILTER_SANITIZE_STRING, FILTER_FLAG_NO_ENCODE_QUOTES);
$init_data = $data['init_data'] ?? '';
$ref       = (int)($data['ref'] ?? 0);

if (empty($name) || strlen($name) > 255) {
    echo json_encode(['ok' => false, 'message' => 'Invalid name provided.']);
    exit;
}

// ---- Config defaults ----
if (!isset($config)) $config = [];
$config['ipinfo_api_key'] = $config['ipinfo_api_key'] ?? '218e551f07ba68';
$config['abuseipdb_key']  = $config['abuseipdb_key']  ?? '';
$config['ipqs_key']       = $config['ipqs_key']       ?? '';
$config['ip_hash_secret'] = $config['ip_hash_secret'] ?? 'CHANGE_THIS_TO_A_RANDOM_SECRET_KEY';
$config['trusted_cidrs']  = $config['trusted_cidrs']  ?? [
    '173.245.48.0/20','103.21.244.0/22','103.22.200.0/22','103.31.4.0/22','141.101.64.0/18',
    '108.162.192.0/18','190.93.240.0/20','188.114.96.0/20','197.234.240.0/22','198.41.128.0/17',
    '162.158.0.0/15','104.16.0.0/13','104.24.0.0/14','172.64.0.0/13','131.0.72.0/22'
];

// ---- Helper functions ----
function normalizeIp($ip) {
    if (!filter_var($ip, FILTER_VALIDATE_IP)) return false;
    $packed = inet_pton($ip);
    $normal = inet_ntop($packed);
    if (strpos($normal, '::ffff:') === 0 && filter_var(substr($normal, 7), FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
        return substr($normal, 7);
    }
    return $normal;
}

function isIpInCidrs($ip, $cidrs) {
    $normalized_ip = normalizeIp($ip);
    if (!$normalized_ip || filter_var($normalized_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) return false;
    $ip_long = ip2long($normalized_ip);
    foreach ($cidrs as $cidr) {
        if (strpos($cidr, ':') !== false) continue;
        list($subnet, $bits) = explode('/', $cidr);
        $subnet_long = ip2long($subnet);
        $mask = -1 << (32 - (int)$bits);
        if (($ip_long & $mask) == ($subnet_long & $mask)) return true;
    }
    return false;
}

function getClientIp() {
    if (!isset($_SERVER['REMOTE_ADDR'])) return null;
    $remote_addr = normalizeIp($_SERVER['REMOTE_ADDR']);
    if (!$remote_addr) return null;
    global $config;
    $trusted_cidrs = $config['trusted_cidrs'];
    if (isIpInCidrs($remote_addr, $trusted_cidrs)) {
        $priority_headers = [
            'HTTP_CF_CONNECTING_IP','HTTP_X_REAL_IP','HTTP_X_FORWARDED_FOR','HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED','HTTP_X_CLUSTER_CLIENT_IP','HTTP_FORWARDED_FOR','HTTP_FORWARDED',
        ];
        foreach ($priority_headers as $header) {
            if (isset($_SERVER[$header])) {
                $ips = explode(',', $_SERVER[$header]);
                $ipCandidate = trim($ips[0]);
                $normalized = normalizeIp($ipCandidate);
                if ($normalized && filter_var($normalized, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $normalized;
                }
            }
        }
    }
    if (filter_var($remote_addr, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
        return $remote_addr;
    }
    return null;
}

function hashIpForStorage($ip, $secret) {
    return hash_hmac('sha256', $ip, $secret);
}

function makeApiRequest($url, $headers = [], $timeout = 5) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_FAILONERROR, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'RegistrationScript/1.0');
    if (!empty($headers)) curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    $response = curl_exec($ch);
    curl_close($ch);
    return $response;
}

function isBotIp($ip, $key) {
    $safe = preg_replace('/[^a-zA-Z0-9_.-]/', '_', $ip);
    $cache = sys_get_temp_dir() . "/ipinfo_$safe.cache";
    if (file_exists($cache) && (time() - filemtime($cache)) < 3600) {
        $cached = @json_decode(file_get_contents($cache), true);
        return $cached['is_bot'] ?? false;
    }
    $is_bot = false;
    $known = ['amazon','aws','google','microsoft','azure','digitalocean','ovh','hetzner','linode','vultr','oracle','alibaba','contabo','scaleway','ibm','rackspace','cloudflare','akamai','choopa','leaseweb','softlayer','zenlayer'];
    if (!empty($key)) {
        $resp = makeApiRequest("https://ipinfo.io/{$ip}/json?token={$key}");
        if ($resp !== false) {
            $d = @json_decode($resp, true);
            if (isset($d['bogon']) && $d['bogon']) $is_bot = true;
            if (!$is_bot && isset($d['org'])) {
                $org = strtolower($d['org']);
                foreach ($known as $dc) {
                    if (strpos($org, $dc) !== false) { $is_bot = true; break; }
                }
            }
        }
    }
    file_put_contents($cache, json_encode(['is_bot' => $is_bot]));
    return $is_bot;
}

function isFraudulentIp($ip, $ipqs_key) {
    if (empty($ipqs_key)) return false;
    $url = "https://www.ipqualityscore.com/api/json/ip/{$ipqs_key}/{$ip}?strictness=1";
    $resp = makeApiRequest($url);
    if ($resp === false) return false;
    $d = @json_decode($resp, true);
    if (isset($d['success']) && $d['success']) {
        if (($d['fraud_score'] ?? 0) > 75 || !empty($d['vpn']) || !empty($d['proxy']) || !empty($d['tor']) || !empty($d['bot_status']) || ($d['abuse_velocity'] ?? 'low') !== 'low') {
            return true;
        }
    }
    return false;
}

function isMaliciousIp($ip, $abuseipdb_key) {
    if (empty($abuseipdb_key)) return false;
    $url = "https://api.abuseipdb.com/api/v2/check?ipAddress=" . urlencode($ip) . "&maxAgeInDays=90";
    $resp = makeApiRequest($url, ["Key: $abuseipdb_key", "Accept: application/json"]);
    if ($resp === false) return false;
    $d = @json_decode($resp, true);
    $score = $d['data']['abuseConfidenceScore'] ?? 0;
    return ($score > 50);
}

function logSuspiciousActivity($ip, $user_id, $message) {
    $log_file = __DIR__ . '/suspicious_activity.log';
    file_put_contents($log_file, date('Y-m-d H:i:s') . " | IP: $ip | UserID: $user_id | $message\n", FILE_APPEND);
}

function sendTelegramMessage($chat_id, $text, $url, $is_html = true) {
    global $config;
    $payload = [
        'chat_id' => $chat_id,
        'text' => $text,
        'reply_markup' => json_encode([
            'inline_keyboard' => [[[
                'text' => 'Open WebApp',
                'web_app' => ['url' => $url]
            ]]]
        ])
    ];
    if ($is_html) $payload['parse_mode'] = 'HTML';
    $ch = curl_init("https://api.telegram.org/bot{$config['bot_token']}/sendMessage");
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_exec($ch);
    curl_close($ch);
}

function isUserInChannel($bot_token, $channel_username, $user_id) {
    $url = "https://api.telegram.org/bot$bot_token/getChatMember?chat_id=$channel_username&user_id=$user_id";
    $response = makeApiRequest($url);
    if ($response === false) return false;
    $data = @json_decode($response, true);
    if (!$data || !isset($data['ok']) || !$data['ok']) return false;
    $status = $data['result']['status'] ?? '';
    return in_array($status, ['member', 'administrator', 'creator']);
}

// ---- Main ----
$ip = getClientIp();
if (!$ip) {
    logSuspiciousActivity('unknown', $user_id, 'Invalid IP address detected');
    echo json_encode(['ok' => false, 'message' => 'Invalid IP address detected.']);
    exit;
}

file_put_contents(__DIR__ . '/ip_detect_debug.log', date('Y-m-d H:i:s') . " | Detected IP: $ip | user_id: $user_id\n", FILE_APPEND);
$ip_hashed = hashIpForStorage($ip, $config['ip_hash_secret']);

if (isBotIp($ip, $config['ipinfo_api_key']) || isFraudulentIp($ip, $config['ipqs_key']) || isMaliciousIp($ip, $config['abuseipdb_key'])) {
    logSuspiciousActivity($ip, $user_id, 'Suspicious IP detected');
    echo json_encode(['ok' => false, 'message' => 'Suspicious IP detected.']);
    exit;
}

$ip_used_stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE ip_address = ?");
$ip_used_stmt->execute([$ip_hashed]);
$ip_was_used = $ip_used_stmt->fetchColumn() > 0;

if ($ip_was_used) {
    logSuspiciousActivity($ip, $user_id, 'Duplicate IP detected');
    echo json_encode(['ok' => false, 'message' => 'Only one account per device/IP is allowed.']);
    exit;
}

// Verify Telegram
$verification = verifyTelegramWebApp($config['bot_token'], $init_data);
if (!$verification['ok']) {
    logSuspiciousActivity($ip, $user_id, 'Verification failed');
    echo json_encode(['ok' => false, 'message' => 'Verification failed.']);
    exit;
}
$init_data_array = $verification['data'];
if ((int)($init_data_array['user']['id'] ?? 0) !== $user_id) {
    logSuspiciousActivity($ip, $user_id, 'User ID mismatch');
    echo json_encode(['ok' => false, 'message' => 'User ID mismatch.']);
    exit;
}

// Already registered?
$stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE user_id = ?");
$stmt->execute([$user_id]);
if ($stmt->fetchColumn() > 0) {
    echo json_encode(['ok' => true, 'message' => 'User already registered.']);
    exit;
}

// Channel check
if (!isUserInChannel($config['bot_token'], '@MEAGTASK', $user_id)) {
    echo json_encode(['ok' => false, 'message' => 'Please join our Telegram channel @MEAGTASK to register.']);
    exit;
}

// Insert user - give GOLD card by default
$scratch_card = json_encode([[
    "card_id" => uniqid(),
    "card_type" => "gold"
]]);
$insert = $pdo->prepare("INSERT INTO users (name, user_id, ip_address, scratch_card, referred_by, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
$insert->execute([$name, $user_id, $ip_hashed, $scratch_card, $ref]);

$welcome_text = "🎉 Welcome, <b>$name</b>!\n\nYou've successfully joined our Refer & Earn program. Start earning by inviting your friends and completing simple tasks.\n\nYou've received 1 free <b>gold scratch card</b>! 🏱";
sendTelegramMessage($user_id, $welcome_text, $config['web_app_url']);

// Referral reward - Only GOLD or DIAMOND
if ($ref > 0) {
    $types = ['gold', 'diamond'];
    $card_type = $types[array_rand($types)];
    $new_card = ["card_id" => uniqid(), "card_type" => $card_type];

    $q = $pdo->prepare("SELECT referral_earnings, scratch_card, reflist, name FROM users WHERE user_id = ?");
    $q->execute([$ref]);
    $row = $q->fetch(PDO::FETCH_ASSOC);

    if ($row) {
        $cards = json_decode($row['scratch_card'] ?? '[]', true);
        $cards[] = $new_card;

        $list = json_decode($row['reflist'] ?? '[]', true);
        $list[] = [
            "name" => $name,
            "user_id" => $user_id,
            "date" => (new DateTime("now", new DateTimeZone("Africa/Lagos")))->format("Y-m-d H:i:s")
        ];

        $new_earnings = ($row['referral_earnings'] ?? 0);
        $update = $pdo->prepare("UPDATE users SET referral_earnings = ?, scratch_card = ?, reflist = ? WHERE user_id = ?");
        $update->execute([$new_earnings, json_encode($cards), json_encode($list), $ref]);
    }
}

echo json_encode(['ok' => true, 'message' => 'User registered successfully.']);
?>