<?php
// Database Configuration
$source_db = [
    'host' => 'localhost',
    'name' => 'sbabewrx_scratch',
    'user' => 'sbabewrx_scratch',
    'pass' => 'sbabewrx_scratch'
];

$target_db = [
    'host' => 'localhost',
    'name' => 'target_database', 
    'user' => 'target_user',
    'pass' => 'target_password',
    'api_url' => 'https://dollar.tgbro.link/sql.php' // URL to your basic API
];

// Set headers for live output
header('Content-Type: text/html; charset=utf-8');
ob_implicit_flush(true);
ob_end_flush();

function sendToAPI($api_url, $query, $params = []) {
    $data = [
        'query' => $query,
        'params' => $params
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json'
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return ['code' => $httpCode, 'response' => json_decode($response, true)];
}

function showProgress($message, $type = 'info') {
    $color = 'black';
    switch($type) {
        case 'success': $color = 'green'; break;
        case 'error': $color = 'red'; break;
        case 'warning': $color = 'orange'; break;
        case 'info': $color = 'blue'; break;
    }
    
    echo "<div style='color: $color; margin: 5px 0;'>" . date('H:i:s') . " - $message</div>";
    flush();
    ob_flush();
    usleep(100000); // Small delay for better visibility
}

// HTML header with auto-refresh
?>
<!DOCTYPE html>
<html>
<head>
    <title>Database Migration Progress</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .progress { background: #f0f0f0; padding: 10px; margin: 10px 0; }
        .success { color: green; }
        .error { color: red; }
        .warning { color: orange; }
        .info { color: blue; }
    </style>
</head>
<body>
    <h1>Database Migration in Progress...</h1>
    <div id="progress-container">
<?php

try {
    // Connect to source database
    showProgress("Connecting to source database...");
    $source_pdo = new PDO(
        "mysql:host={$source_db['host']};dbname={$source_db['name']};charset=utf8",
        $source_db['user'],
        $source_db['pass']
    );
    $source_pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    showProgress("✓ Connected to source database", 'success');

    // Test target API connection
    showProgress("Testing target API connection...");
    $test = sendToAPI($target_db['api_url'], 'SELECT 1 as test');
    if ($test['code'] !== 200) {
        throw new Exception("Target API not accessible");
    }
    showProgress("✓ Target API is accessible", 'success');

    // Get all tables from source database
    showProgress("Fetching table list from source...");
    $tables_stmt = $source_pdo->query("SHOW TABLES");
    $tables = $tables_stmt->fetchAll(PDO::FETCH_COLUMN);
    showProgress("Found " . count($tables) . " tables: " . implode(', ', $tables), 'success');

    $total_tables = count($tables);
    $processed_tables = 0;

    foreach ($tables as $table) {
        $processed_tables++;
        showProgress("Processing table {$processed_tables}/{$total_tables}: {$table}");
        
        // Step 1: Get table structure
        showProgress("Getting structure for table: {$table}");
        $create_stmt = $source_pdo->query("SHOW CREATE TABLE `{$table}`");
        $create_table = $create_stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!isset($create_table['Create Table'])) {
            showProgress("✗ Could not get structure for table: {$table}", 'error');
            continue;
        }

        $create_sql = $create_table['Create Table'];
        
        // Step 2: Create table in target database
        showProgress("Creating table in target: {$table}");
        $result = sendToAPI($target_db['api_url'], "DROP TABLE IF EXISTS `{$table}`");
        $result = sendToAPI($target_db['api_url'], $create_sql);
        
        if (!$result['response']['success']) {
            showProgress("✗ Failed to create table: {$table}", 'error');
            continue;
        }
        showProgress("✓ Table created: {$table}", 'success');

        // Step 3: Get data from source table
        showProgress("Fetching data from source: {$table}");
        $data_stmt = $source_pdo->query("SELECT * FROM `{$table}`");
        $total_rows = $source_pdo->query("SELECT COUNT(*) FROM `{$table}`")->fetchColumn();
        
        showProgress("Found {$total_rows} rows in table: {$table}");
        
        if ($total_rows > 0) {
            // Get column names
            $columns = [];
            for ($i = 0; $i < $data_stmt->columnCount(); $i++) {
                $col = $data_stmt->getColumnMeta($i);
                $columns[] = $col['name'];
            }
            
            $column_list = '`' . implode('`, `', $columns) . '`';
            $placeholders = ':' . implode(', :', $columns);
            
            $inserted_rows = 0;
            $batch_size = 100;
            $batch_count = 0;
            
            // Process data in batches
            while ($row = $data_stmt->fetch(PDO::FETCH_ASSOC)) {
                $insert_query = "INSERT INTO `{$table}` ({$column_list}) VALUES ({$placeholders})";
                
                $result = sendToAPI($target_db['api_url'], $insert_query, $row);
                
                if ($result['response']['success']) {
                    $inserted_rows++;
                } else {
                    showProgress("✗ Failed to insert row: " . $result['response']['error'], 'error');
                }
                
                $batch_count++;
                
                // Show progress every batch
                if ($batch_count % $batch_size === 0) {
                    $percent = round(($inserted_rows / $total_rows) * 100, 2);
                    showProgress("Progress for {$table}: {$inserted_rows}/{$total_rows} ({$percent}%)");
                }
            }
            
            showProgress("✓ Completed table {$table}: {$inserted_rows}/{$total_rows} rows migrated", 'success');
        } else {
            showProgress("✓ Table {$table} is empty, no data to migrate", 'warning');
        }
        
        // Progress for overall migration
        $overall_percent = round(($processed_tables / $total_tables) * 100, 2);
        showProgress("Overall progress: {$processed_tables}/{$total_tables} tables ({$overall_percent}%)", 'info');
    }
    
    showProgress("🎉 Database migration completed successfully!", 'success');
    showProgress("Migrated {$total_tables} tables with data", 'success');
    
} catch (Exception $e) {
    showProgress("✗ Error: " . $e->getMessage(), 'error');
}

?>
    </div>
    <h2>Migration Completed</h2>
    <p>Check the progress log above for details.</p>
</body>
</html>