<?php
require 'config.php';
header('Content-Type: application/json');

function log_error($message) {
    error_log("[Draw API] " . $message);
}

try {
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $user_id  = $_GET['user_id'] ?? null;
    $action   = $_GET['action'] ?? null;
    $tickets  = isset($_GET['tickets']) ? intval($_GET['tickets']) : 0;
    $username = $_GET['username'] ?? null;

    if (!$user_id) {
        echo json_encode(['ok' => false, 'message' => 'User ID is required']);
        exit;
    }

    // Log inputs for debugging
    log_error("Received: user_id=$user_id, action=$action, tickets=$tickets, username=" . var_export($username, true));

    // ====== DETAILS ======
    if ($action === 'details') {
        try {
            // Total tickets
            $stmt = $pdo->query("SELECT COALESCE(SUM(tickets),0) FROM draw_entries");
            $total = $stmt->fetchColumn();

            // User tickets
            $stmt = $pdo->prepare("SELECT COALESCE(SUM(tickets),0) FROM draw_entries WHERE user_id = ?");
            $stmt->execute([$user_id]);
            $user_tickets = $stmt->fetchColumn();

            // User points
            $stmt = $pdo->prepare("SELECT ticket_point FROM users WHERE user_id = ?");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            $user_points = $user ? (int)$user['ticket_point'] : 0;

            $winning_chance = $total > 0 ? round(($user_tickets / $total) * 100, 2) . "%" : "0.00%";

            // Last 5 winners
            $stmt = $pdo->query("SELECT winner_name FROM draws WHERE winner_name IS NOT NULL ORDER BY id DESC LIMIT 5");
            $winners = $stmt->fetchAll(PDO::FETCH_COLUMN);
            $winners_response = $winners ?: ["No recent winners"];

            echo json_encode([
                'ok' => true,
                'user_tickets' => (int)$user_tickets,
                'user_points' => $user_points,
                'total_tickets' => (int)$total,
                'winning_chance' => $winning_chance,
                'last_winners' => $winners_response,
                'prize_per_winner' => 100000
            ]);
        } catch (Exception $e) {
            log_error("DETAILS error: " . $e->getMessage());
            echo json_encode(['ok' => false, 'message' => 'Failed to load details']);
        }
        exit;
    }

    // ====== PARTICIPATE ======
    if ($action === 'participate') {
        if ($tickets <= 0 || $tickets > 10000) {
            echo json_encode(['ok' => false, 'message' => 'Ticket amount must be between 1 and 10,000']);
            exit;
        }

        try {
            // Start transaction
            $pdo->beginTransaction();

            // Calculate required points (1 ticket = 30 points)
            $required_points = $tickets * 30;

            // Verify user exists and has enough points
            $stmt = $pdo->prepare("SELECT ticket_point FROM users WHERE user_id = ? FOR UPDATE");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$user) {
                $pdo->rollBack();
                echo json_encode(['ok' => false, 'message' => 'User not found']);
                exit;
            }

            if ($user['ticket_point'] < $required_points) {
                $pdo->rollBack();
                echo json_encode(['ok' => false, 'message' => "Not enough points. Required: $required_points, Available: {$user['ticket_point']}"]);
                exit;
            }

            // Deduct points from user
            $stmt = $pdo->prepare("UPDATE users SET ticket_point = ticket_point - ? WHERE user_id = ?");
            $stmt->execute([$required_points, $user_id]);
            $affected_rows = $stmt->rowCount();
            log_error("Deducted $required_points points ($tickets tickets) for user_id=$user_id, affected rows: $affected_rows");

            // Determine username: Use provided username, or from draw_entries, or user_id
            $final_username = $username;
            if (empty($final_username)) {
                $stmt = $pdo->prepare("SELECT username FROM draw_entries WHERE user_id = ? LIMIT 1");
                $stmt->execute([$user_id]);
                $entry = $stmt->fetch(PDO::FETCH_ASSOC);
                $final_username = $entry['username'] ?? (string)$user_id;
            }

            // Validate final username
            if (empty($final_username) || strlen($final_username) > 100) {
                $pdo->rollBack();
                log_error("Invalid final username: " . var_export($final_username, true));
                echo json_encode(['ok' => false, 'message' => 'Username must be non-empty and less than 100 characters']);
                exit;
            }

            // Add entry to draw
            $stmt = $pdo->prepare("INSERT INTO draw_entries (user_id, username, tickets) VALUES (?, ?, ?)");
            $stmt->execute([$user_id, $final_username, $tickets]);

            // Check total tickets
            $stmt = $pdo->query("SELECT COALESCE(SUM(tickets),0) FROM draw_entries");
            $total = $stmt->fetchColumn();

            if ($total >= 5000) {
                // Lock draw_entries table
                $stmt = $pdo->query("SELECT user_id, username FROM draw_entries FOR UPDATE");
                $entries = $stmt->fetchAll(PDO::FETCH_ASSOC);

                if ($entries) {
                    // Pick 5 random winners
                    $winners = [];
                    $reward = 100000; // 100k MEGA

                    $entryPool = $entries;
                    for ($i = 0; $i < 5; $i++) {
                        if (count($entryPool) === 0) break;
                        $randomIndex = array_rand($entryPool);
                        $winner = $entryPool[$randomIndex];
                        unset($entryPool[$randomIndex]);

                        // Pay winner
                        $stmt = $pdo->prepare("UPDATE users SET balance = balance + ? WHERE user_id = ?");
                        $stmt->execute([$reward, $winner['user_id']]);

                        // Save to draws history
                        $stmt = $pdo->prepare("INSERT INTO draws (user_id, winner_name, reward) VALUES (?, ?, ?)");
                        $stmt->execute([$winner['user_id'], $winner['username'], $reward]);

                        $winners[] = [
                            'username' => $winner['username'],
                            'reward' => $reward
                        ];
                    }

                    // Log winners
                    log_error("Draw completed, winners: " . json_encode($winners));

                    // Reset draw (ensures tickets are not refunded)
                    $pdo->query("TRUNCATE TABLE draw_entries");

                    $pdo->commit();
                    echo json_encode([
                        'ok' => true,
                        'message' => 'Draw completed',
                        'winners' => $winners,
                        'prize_per_winner' => $reward
                    ]);
                    exit;
                }
            }

            $pdo->commit();
            echo json_encode([
                'ok' => true,
                'message' => 'Participation successful',
                'data' => [
                    'tickets_entered' => $tickets,
                    'points_deducted' => $required_points,
                    'new_point_balance' => $user['ticket_point'] - $required_points
                ]
            ]);
        } catch (Exception $e) {
            $pdo->rollBack();
            log_error("PARTICIPATE error: " . $e->getMessage());
            echo json_encode(['ok' => false, 'message' => 'Failed to participate: ' . $e->getMessage()]);
        }
        exit;
    }

    // ====== INVALID ACTION ======
    echo json_encode(['ok' => false, 'message' => 'Invalid action']);

} catch (Exception $e) {
    log_error("GENERAL error: " . $e->getMessage());
    echo json_encode(['ok' => false, 'message' => 'Server error']);
}