<?php
require 'config.php';

// Check if the configuration is available
if (!isset($config['bot_token'])) {
    error_log("FATAL ERROR: bot_token is missing in config.php");
    echo json_encode(['ok' => false, 'message' => 'Server configuration error.']);
    exit;
}

// ===== LOG SETUP =====
ini_set('log_errors', 1);
// Changed log file name for clarity
ini_set('error_log', __DIR__ . '/star_invoice.log');

// Read input from POST body
$input = file_get_contents("php://input");
$data = json_decode($input, true) ?? [];
$action = $data['action'] ?? 'create_invoice'; // Assume action is create_invoice
$user_id = $data['user_id'] ?? null;

// Use the correct input variable name 'star_amount' from the frontend
$star_amount_to_buy = (int) ($data['star_amount'] ?? 0); 
$init_data = $data['init_data'] ?? null;

error_log("=== Star Invoice Request ===");
error_log("Input: " . $input);
error_log("Action: " . $action . " | User: " . $user_id . " | Amount: " . $star_amount_to_buy . " | Init Data: " . substr($init_data ?? '', 0, 50) . "...");

// --- REVISED LOGIC ---

// ✅ Validation
$min_amount = 100; // Match the minimum from your frontend
if ($action !== 'create_invoice' || !$user_id || $star_amount_to_buy < $min_amount) {
    $msg = "Invalid request. Action must be 'create_invoice', User ID required, and Star amount must be at least $min_amount.";
    error_log("ERROR: " . $msg);
    echo json_encode(['ok' => false, 'message' => $msg]);
    exit;
}

// --- 🔑 CORE CHANGE: Price in XTR (Telegram Stars) equals the Stars the user buys
// If user wants to buy 100 Stars, the price is 100 XTR.
$star_price_in_xtr = $star_amount_to_buy; 
error_log("Price => $star_price_in_xtr XTR for $star_amount_to_buy Stars");

// ✅ Payload passed to callback later (important for crediting after payment)
$payload = $user_id . "|" . $star_amount_to_buy . "|" . time();

$invoice = [
    'title' => "Purchase {$star_amount_to_buy} Stars",
    'description' => "Purchase {$star_amount_to_buy} Stars for use in Win NFT game.",
    'payload' => $payload,
    'currency' => "XTR",
    'prices' => [
        // Display the price as the amount of Stars the user is buying
        ["label" => "{$star_amount_to_buy} Stars", "amount" => $star_price_in_xtr]
    ]
];

// ✅ Request to Telegram
$url = "https://api.telegram.org/bot{$config['bot_token']}/createInvoiceLink";
$ch = curl_init($url);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($invoice));
$response = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

error_log("Telegram Response (HTTP $http_code): " . $response);

$result = json_decode($response, true);

if (!empty($result['ok']) && isset($result['result'])) {
    echo json_encode([
        'ok' => true,
        'invoice_link' => $result['result']
    ]);
} else {
    $desc = $result['description'] ?? 'Unknown error';
    error_log("Invoice Error: " . $desc);
    echo json_encode(['ok' => false, 'message' => $desc]);
}

// ===== AUTO-SET WEBHOOK (Required for receiving payment confirmation) =====
$webhook_url = "https://wenepay.cyou/scratch/api/webhook12.php";
$setWebhookUrl = "https://api.telegram.org/bot{$config['bot_token']}/setWebhook?url=" . urlencode($webhook_url);

// Check if the URL is accessible before setting (optional, but good practice)
if (!filter_var($webhook_url, FILTER_VALIDATE_URL)) {
    error_log("WARNING: Invalid Webhook URL format: " . $webhook_url);
} else {
    // Note: Use file_get_contents for simple GET, or curl for POST/more complex needs
    $webhook_response = @file_get_contents($setWebhookUrl);
    if ($webhook_response === FALSE) {
        error_log("WARNING: Failed to connect to Telegram API to set webhook.");
    } else {
        error_log("Webhook Set Response: " . $webhook_response);
    }
}
?>
