<?php
declare(strict_types=1);

require 'config.php';
require 'init_data_check.php';
header('Content-Type: application/json');

// Helper: return JSON
function respond(bool $ok, string $message, array $extra = []): void {
    $res = ['ok' => $ok, 'message' => $message];
    if (!empty($extra)) $res = array_merge($res, $extra);
    echo json_encode($res);
    exit;
}

// Function to fetch Telegram Star Gifts and their prices
function fetchStarGifts(string $botToken): array {
    $url = "https://api.telegram.org/bot{$botToken}/getStarGifts?hash=0";  // Added hash=0
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPGET, true);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    error_log("StarGifts raw response: " . $response);  // Log raw response for debugging
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        error_log("StarGifts decoded data: " . json_encode($data));  // Log decoded data
        
        if ($data['ok']) {
            // Return map of gift ID to stars (price)
            $gifts = [];
            foreach ($data['result']['gifts'] ?? [] as $gift) {
                $gifts[$gift['id']] = $gift['stars'] ?? 0;  // Correct field: 'stars'
            }
            return $gifts;
        }
    }
    // On error, return empty
    error_log("Failed to fetch Star Gifts: HTTP {$httpCode} - " . ($response ?? 'No response'));
    return [];
}

// Input
$input = json_decode(file_get_contents("php://input"), true);
if (!$input || !isset($input['user_id']) || !isset($input['init_data'])) {
    respond(false, "Missing required fields.");
}

$user_id = trim($input['user_id']);
$init_data = $input['init_data'];

// SPIN COST (fixed)
$cost = 200;

// Validate Telegram initData
if ($init_data !== '786786') {
    $verify = verifyTelegramWebApp($config['bot_token'], $init_data);
    if (!$verify['ok']) respond(false, "Init data verification failed.");
    if ((string)$verify['data']['user']['id'] !== $user_id) {
        respond(false, "User ID mismatch.");
    }
}

// Fetch Star Gifts prices once (assuming reward_id matches Telegram gift ID)
$giftPrices = fetchStarGifts($config['bot_token']);

// Simple title-to-price map if IDs don't match (adjust based on your logs/API response)
$titleToPriceMap = [  // Example mappings; replace with actual from logs
    'Cookiehart' => 100,  // Placeholder; get real from API
    'Lollipop' => 150,
    'Desk Calendar' => 200,
    'Coffee Cup' => 250,
    'Champagne' => 500,
    'Rose' => 300,
    // Add more as needed
];

try {
    $pdo->beginTransaction();

    // Fetch user (lock the row)
    $stmt = $pdo->prepare("SELECT stars, balance, spin_count FROM users WHERE user_id = ? FOR UPDATE");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        $pdo->rollBack();
        respond(false, "User not found.");
    }

    $user['stars'] = (float)$user['stars'];
    $user['balance'] = (float)($user['balance'] ?? 0);

    // Check star balance
    if ($user['stars'] < $cost) {
        $pdo->rollBack();
        respond(false, "Not enough stars to spin.", ['user' => $user]);
    }

    // Deduct the cost
    $new_stars = $user['stars'] - $cost;
    $stmt = $pdo->prepare("UPDATE users SET stars = ? WHERE user_id = ?");
    $stmt->execute([$new_stars, $user_id]);
    $current_stars = $new_stars;

    // Fetch configured rewards
    $stmt = $pdo->prepare("
        SELECT r.id, r.title, r.type, r.value, r.img, p.prob
        FROM rewards r
        JOIN probabilities p ON r.id = p.reward_id
        WHERE p.cost = 200
    ");
    $stmt->execute();
    $rewards = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($rewards)) {
        $pdo->rollBack();
        respond(false, "Spin rewards not configured.");
    }

    /*
     * REMOVE 100k STAR & 200k STAR (completely impossible)
     */
    $filtered = [];
    foreach ($rewards as $r) {
        if ($r['type'] === 'star' && ($r['value'] == 100000 || $r['value'] == 200000)) {
            continue; // skip these star rewards
        }
        $filtered[] = $r;
    }
    $rewards = $filtered;

    /*
     * ADD SUPER RARE MEGA JACKPOTS
     */
    $superRareMega = [
        [
            'id' => 'mega_100k',
            'title' => '100,000 Mega',
            'type' => 'mega',
            'value' => 100000,
            'img' => 'mega.png',
            'prob' => 0.00005 // 0.005% chance (1 in 20,000)
        ],
        [
            'id' => 'mega_200k',
            'title' => '200,000 Mega',
            'type' => 'mega',
            'value' => 200000,
            'img' => 'mega.png',
            'prob' => 0.00001 // 0.001% chance (1 in 100,000)
        ]
    ];

    // Merge jackpot list into rewards
    $rewards = array_merge($rewards, $superRareMega);

    // Weighted random selection
    $rand = mt_rand(1, 1000000) / 1000000; // more precision
    $cumulative = 0;
    $winning_reward = $rewards[0];

    foreach ($rewards as $r) {
        $cumulative += (float)$r['prob'];
        if ($rand <= $cumulative) {
            $winning_reward = $r;
            break;
        }
    }

    // If it's a gift, add price from Telegram API (try ID first, fallback to title map if no match)
    if ($winning_reward['type'] === 'gift') {
        $giftId = $winning_reward['id'];
        $winning_reward['price'] = $giftPrices[$giftId] ?? $titleToPriceMap[$winning_reward['title']] ?? 0;  // Fallback to title map
    }

    // Apply reward
    switch ($winning_reward['type']) {

        case 'star':
            $reward_value = (float)$winning_reward['value'];
            $final_stars = $current_stars + $reward_value;

            $stmt = $pdo->prepare("UPDATE users SET stars = ? WHERE user_id = ?");
            $stmt->execute([$final_stars, $user_id]);

            $message = "You won {$reward_value} stars!";
            break;

        case 'mega':
            $reward_value = (float)$winning_reward['value'];
            $final_balance = $user['balance'] + $reward_value;

            $stmt = $pdo->prepare("UPDATE users SET balance = ? WHERE user_id = ?");
            $stmt->execute([$final_balance, $user_id]);

            $message = "You won {$reward_value} Mega!";
            break;

        case 'gift':
            $unlock_at = date('Y-m-d H:i:s', strtotime('+22 days'));

            $stmt = $pdo->prepare("
                INSERT INTO inventory (user_id, reward_id, title, img, unlock_at)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $user_id,
                $winning_reward['id'],
                $winning_reward['title'],
                $winning_reward['img'],
                $unlock_at
            ]);

            $message = "You won a gift: {$winning_reward['title']}!";
            break;

        default:
            $pdo->rollBack();
            respond(false, "Unknown reward type.");
    }

    // Fetch updated user info
    $stmt = $pdo->prepare("SELECT stars, balance, spin_count FROM users WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $updatedUser = $stmt->fetch(PDO::FETCH_ASSOC);

    // Fetch inventory and add prices where applicable
    $stmt = $pdo->prepare("
        SELECT reward_id, title, img, unlock_at
        FROM inventory
        WHERE user_id = ?
        ORDER BY unlock_at DESC
    ");
    $stmt->execute([$user_id]);
    $inventory = $stmt->fetchAll(PDO::FETCH_ASSOC);

    foreach ($inventory as &$item) {
        $giftId = $item['reward_id'];
        $item['price'] = $giftPrices[$giftId] ?? $titleToPriceMap[$item['title']] ?? 0;  // Fallback to title map
    }
    unset($item);  // Unset reference

    $pdo->commit();

    // Full response
    respond(true, $message, [
        'user' => $updatedUser,
        'reward' => $winning_reward,
        'inventory' => $inventory
    ]);

} catch (Exception $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    error_log("Spin Error: " . $e->getMessage());
    respond(false, "Internal error.");
}
?>