<?php
// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Set content type
header('Content-Type: application/json');

// Load config
try {
    require 'config.php';
    require 'init_data_check.php';
} catch (Exception $e) {
    echo json_encode(['ok' => false, 'message' => 'Failed to load file: '.$e->getMessage()]);
    exit;
}

// Read input
$raw_input = file_get_contents("php://input");
if (!$raw_input) {
    echo json_encode(['ok' => false, 'message' => 'No input received.']);
    exit;
}

$data = json_decode($raw_input, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode(['ok' => false, 'message' => 'Invalid JSON: '.json_last_error_msg()]);
    exit;
}

$user_id = (int)($data['user_id'] ?? 0);
$init_data = $data['init_data'] ?? '';
$action = $data['action'] ?? 'fetch';

if ($user_id <= 0) {
    echo json_encode(['ok' => false, 'message' => 'Invalid user_id.']);
    exit;
}
if (empty($init_data)) {
    echo json_encode(['ok' => false, 'message' => 'Missing init_data.']);
    exit;
}

// Verify init_data (unless bypass code)
if ($init_data !== '786786') {
    try {
        $verification = verifyTelegramWebApp($config['bot_token'], $init_data);
        if ((!$verification['ok']) ?? true) {
            echo json_encode(['ok' => false, 'message' => 'Init data verification failed.']);
            exit;
        }
        if ((int)$verification['data']['user']['id'] !== $user_id) {
            echo json_encode(['ok' => false, 'message' => 'User ID mismatch.']);
            exit;
        }
    } catch (Exception $e) {
        echo json_encode(['ok' => false, 'message' => 'Verification failure: ' . $e->getMessage()]);
        exit;
    }
}

// Fetch user info
try {
    $stmt = $pdo->prepare("SELECT user_id, name, balance, reflist, referral_earnings FROM users WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        echo json_encode(['ok' => false, 'message' => 'User not found.']);
        exit;
    }
} catch (PDOException $e) {
    echo json_encode(['ok' => false, 'message' => 'User fetch error: '.$e->getMessage()]);
    exit;
}

// CLAIM earnings
if ($action === 'claim') {
    if ($user['referral_earnings'] > 0) {
        try {
            $pdo->beginTransaction();

            $update = $pdo->prepare("UPDATE users SET balance = balance + referral_earnings, referral_earnings = 0 WHERE user_id = ?");
            $update->execute([$user_id]);

            $pdo->commit();

            // refetch info
            $stmt = $pdo->prepare("SELECT balance, referral_earnings FROM users WHERE user_id = ?");
            $stmt->execute([$user_id]);
            $updated = $stmt->fetch(PDO::FETCH_ASSOC);

            echo json_encode([
                'ok' => true,
                'message' => 'Referral earnings claimed.',
                'new_balance' => (int)$updated['balance'],
                'referral_earnings' => (float)$updated['referral_earnings']
            ]);
            exit;

        } catch (PDOException $e) {
            $pdo->rollBack();
            echo json_encode(['ok' => false, 'message' => 'Claim error: '.$e->getMessage()]);
            exit;
        }
    } else {
        echo json_encode(['ok' => false, 'message' => 'No earnings to claim.']);
        exit;
    }
}

// Decode referral list
$referrals = json_decode($user['reflist'] ?? '[]', true);
if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode(['ok' => false, 'message' => 'Invalid referral list JSON.']);
    exit;
}
$total_refer = count($referrals);

// ================================
// FIXED LEADERBOARD (ACTIVE REFERRALS ONLY)
// ================================
try {
    $top_stmt = $pdo->query("
        SELECT 
            u.user_id,
            u.name,
            COUNT(r.referred_id) AS total_refer
        FROM users u

        -- only users with referrals
        INNER JOIN referrals r 
            ON u.user_id = r.referrer_id

        -- only referrals who watched ads
        INNER JOIN (
            SELECT user_id, COUNT(*) AS ads_watched 
            FROM adswatched 
            GROUP BY user_id
        ) aw 
            ON r.referred_id = aw.user_id

        WHERE aw.ads_watched >= 0

        GROUP BY u.user_id
        HAVING total_refer >= 1
        ORDER BY total_refer DESC
        LIMIT 10
    ");
    $top_referrers = $top_stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    echo json_encode([
        'ok' => false,
        'message' => 'Leaderboard error: ' . $e->getMessage()
    ]);
    exit;
}

// Fetch bonuses
try {
    $bonus_stmt = $pdo->query("SELECT * FROM bonuses LIMIT 1");
    $bonus = $bonus_stmt->fetch(PDO::FETCH_ASSOC) ?: [];
} catch (PDOException $e) {
    echo json_encode(['ok' => false, 'message' => 'Bonus fetch error: '.$e->getMessage()]);
    exit;
}

// Attach leaderboard bonus
foreach ($top_referrers as $i => &$row) {
    $rank = $i+1;
    $key = match($rank) {
        1 => 'first_prize',
        2 => 'second_prize',
        3 => 'third_prize',
        4 => 'fourth_prize',
        5 => 'fifth_prize',
        6 => 'sixth_prize',
        7 => 'seventh_prize',
        8 => 'eighth_prize',
        9 => 'ninth_prize',
        10 => 'tenth_prize',
        default => null
    };

    $row['bonus'] = $bonus[$key] ?? 0;
}

// Fetch bot username
try {
    $s = $pdo->query("SELECT bot_username FROM settings LIMIT 1");
    $setting = $s->fetch(PDO::FETCH_ASSOC);
    $bot_username = $setting['bot_username'] ?? '@diamond_card_bot';
} catch (PDOException $e) {
    $bot_username = '@diamond_card_bot';
}

// Final output
echo json_encode([
    'ok' => true,
    'user_details' => [
        'user_id' => $user['user_id'],
        'name' => $user['name'],
        'balance' => (int)$user['balance'],
        'total_refer' => $total_refer,
        'referral_earnings' => (float)$user['referral_earnings'],
        'bot_username' => $bot_username
    ],
    'referrals' => $referrals,
    'top_referrers' => $top_referrers
], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

?>