<?php
require_once 'config.php'; // Include your config file for DB connection

// Secret key from the postback instructions
$secret_key = 'c7e76ce9971419e2494ccd2edd488a8a';

// Whitelisted IPs
$whitelisted_ips = ['51.81.120.73', '142.111.248.18'];

// For demo purposes: We'll assume a 'users' table with 'user_id' and 'balance' columns (integer balance for simplicity)
// And a 'transactions' table to track processed transaction_ids (id, transaction_id unique)

// Function to verify hash
function verify_hash($user_id, $revenue, $provided_hash, $secret_key) {
    $to_hash = strval($user_id) . strval($revenue) . $secret_key;
    $computed_hash = hash('sha256', $to_hash);
    return $computed_hash === $provided_hash;
}

// Set JSON header for all responses
header('Content-Type: application/json');

// Only process if it's a GET request
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    // Extract parameters
    $user_id = isset($_GET['userID']) ? $_GET['userID'] : null;
    $transaction_id = isset($_GET['transactionID']) ? $_GET['transactionID'] : null;
    $revenue = isset($_GET['revenue']) ? $_GET['revenue'] : null; // String like '1.35' or '-1.35'
    $currency_amount = isset($_GET['currencyAmount']) ? $_GET['currencyAmount'] : null; // String like '83125' or '-83125'
    $provided_hash = isset($_GET['hash']) ? $_GET['hash'] : null; // Optional
    $ip_param = isset($_GET['ip']) ? $_GET['ip'] : null; // Optional
    $type = isset($_GET['type']) ? $_GET['type'] : null; // Optional: 'credit' or 'chargeback'
    $withdraw_id = isset($_GET['withdrawid']) ? $_GET['withdrawid'] : null; // Optional

    // Basic validation: required params
    if (!$user_id || !$transaction_id || !$revenue || !$currency_amount) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Missing required parameters']);
        exit;
    }

    // Security check: IP whitelisting or hash verification
    $remote_ip = $_SERVER['REMOTE_ADDR'];
    $ip_verified = in_array($remote_ip, $whitelisted_ips);
    $hash_verified = $provided_hash && verify_hash($user_id, $revenue, $provided_hash, $secret_key);

    if (!$ip_verified && !$hash_verified) {
        http_response_code(403);
        echo json_encode(['status' => 'error', 'message' => 'Unauthorized: Invalid IP or hash']);
        exit;
    }

    // If DB connection failed, log and fail gracefully (but try to proceed if possible; here we need DB)
    if ($pdo === null) {
        error_log("Postback failed due to DB connection issue.");
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Internal server error']);
        exit;
    }

    // Check if transaction already processed
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM transactions WHERE transaction_id = :transaction_id");
    $stmt->execute(['transaction_id' => $transaction_id]);
    if ($stmt->fetchColumn() > 0) {
        http_response_code(200);
        echo json_encode(['status' => 'success', 'message' => 'Transaction already processed']);
        exit;
    }

    // Parse amounts (handle negatives for chargebacks)
    $revenue_float = filter_var($revenue, FILTER_VALIDATE_FLOAT);
    $currency_int = filter_var($currency_amount, FILTER_VALIDATE_INT);
    if ($revenue_float === false || $currency_int === false) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Invalid amount formats']);
        exit;
    }

    // Start transaction for atomicity
    $pdo->beginTransaction();

    try {
        // Get current balance or initialize
        $stmt = $pdo->prepare("SELECT balance FROM users WHERE user_id = :user_id FOR UPDATE");
        $stmt->execute(['user_id' => $user_id]);
        $row = $stmt->fetch();
        if ($row) {
            $current_balance = $row['balance'];
        } else {
            // Insert new user if not exists (assuming balance starts at 0)
            $stmt = $pdo->prepare("INSERT INTO users (user_id, balance) VALUES (:user_id, 0)");
            $stmt->execute(['user_id' => $user_id]);
            $current_balance = 0;
        }

        // Update balance (adds reward for credits, subtracts for chargebacks via negative amounts)
        $new_balance = $current_balance + $currency_int;
        $stmt = $pdo->prepare("UPDATE users SET balance = :new_balance WHERE user_id = :user_id");
        $stmt->execute(['new_balance' => $new_balance, 'user_id' => $user_id]);

        // Record transaction
        $stmt = $pdo->prepare("INSERT INTO transactions (transaction_id, user_id, revenue, currency_amount, type, withdraw_id) 
                               VALUES (:transaction_id, :user_id, :revenue, :currency_amount, :type, :withdraw_id)");
        $stmt->execute([
            'transaction_id' => $transaction_id,
            'user_id' => $user_id,
            'revenue' => $revenue_float,
            'currency_amount' => $currency_int,
            'type' => $type,
            'withdraw_id' => $withdraw_id
        ]);

        $pdo->commit();

        // For logging (optional)
        error_log("Processed $type for user $user_id: revenue=$revenue_float, currency=$currency_int");

        // Return success with new balance for better responsiveness
        http_response_code(200);
        echo json_encode(['status' => 'success', 'message' => 'OK', 'new_balance' => $new_balance]);
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Postback processing failed: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Internal server error']);
    }
} else {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
}
?>