<?php
require 'config.php';

// ===== LOGGING SETUP =====
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/payment-callback.log');

// ===== 1. READ TELEGRAM UPDATE =====
$input = file_get_contents('php://input');
$update = json_decode($input, true) ?? [];

error_log("=== Payment Callback at " . date('Y-m-d H:i:s') . " ===");
error_log("Raw Input: " . $input);

// ===== 2. HANDLE PRE_CHECKOUT_QUERY =====
if (isset($update['pre_checkout_query'])) {
    $pre_checkout_query = $update['pre_checkout_query'];
    $query_id = $pre_checkout_query['id'];
    $user_id = $pre_checkout_query['from']['id'];
    $payload = $pre_checkout_query['invoice_payload'];

    error_log("PreCheckoutQuery => query_id: $query_id, user_id: $user_id, payload: $payload");

    // Validate payload
    $payload_parts = explode('|', $payload);
    if (count($payload_parts) !== 3) {
        error_log("ERROR: Invalid payload format");
        sendPreCheckoutResponse($config['bot_token'], $query_id, false, "Invalid payload");
        exit;
    }

    list($payload_user_id, $credits, $timestamp) = $payload_parts;
    $credits = (int) $credits;
    $timestamp = (int) $timestamp;

    // Validate payload data
    if ($payload_user_id != $user_id || $credits < 80 || $credits % 80 !== 0) {
        error_log("ERROR: Invalid payload data (user_id: $payload_user_id, credits: $credits)");
        sendPreCheckoutResponse($config['bot_token'], $query_id, false, "Invalid payment data");
        exit;
    }

    // Check timestamp (e.g., allow payments within 1 hour)
    if (time() - $timestamp > 3600) {
        error_log("ERROR: Payment request expired");
        sendPreCheckoutResponse($config['bot_token'], $query_id, false, "Payment request expired");
        exit;
    }

    // Approve the pre-checkout query
    sendPreCheckoutResponse($config['bot_token'], $query_id, true);
    exit;
}

// ===== 3. HANDLE SUCCESSFUL PAYMENT =====
if (isset($update['message']['successful_payment'])) {
    $payment = $update['message']['successful_payment'];
    $user_id = $update['message']['from']['id'];
    $payload = $payment['invoice_payload'];
    $star_amount = (int) $payment['total_amount'];

    error_log("SuccessfulPayment => user_id: $user_id, payload: $payload, star_amount: $star_amount");

    // Parse payload
    $payload_parts = explode('|', $payload);
    if (count($payload_parts) !== 3) {
        error_log("ERROR: Invalid payload format");
        exit;
    }

    list($payload_user_id, $credits, $timestamp) = $payload_parts;
    $credits = (int) $credits;
    $timestamp = (int) $timestamp;

    // Validate payload data
    if ($payload_user_id != $user_id || $credits < 80 || $credits % 80 !== 0) {
        error_log("ERROR: Invalid payload data (user_id: $payload_user_id, credits: $credits)");
        exit;
    }

    // Verify star amount
    $expected_star_cost = (int) floor(($credits / 80) * 100);
    if ($star_amount !== $expected_star_cost) {
        error_log("ERROR: Invalid star amount: expected $expected_star_cost, received $star_amount");
        exit;
    }

    // Update user's credits in the database
    global $conn;
    try {
        $stmt = $conn->prepare("UPDATE users SET credits = credits + ? WHERE user_id = ?");
        $stmt->bind_param('is', $credits, $user_id);
        $stmt->execute();

        if ($stmt->affected_rows > 0) {
            error_log("SUCCESS: Added $credits credits to user $user_id");
        } else {
            // User might not exist, insert new record
            $stmt = $conn->prepare("INSERT INTO users (user_id, credits) VALUES (?, ?) ON DUPLICATE KEY UPDATE credits = credits + ?");
            $stmt->bind_param('sii', $user_id, $credits, $credits);
            $stmt->execute();
            error_log("SUCCESS: Inserted/Updated $credits credits for user $user_id");
        }
        $stmt->close();
    } catch (Exception $e) {
        error_log("ERROR: Database update failed: " . $e->getMessage());
        exit;
    }

    // Optionally notify the user
    sendMessage($config['bot_token'], $user_id, "Successfully purchased $credits credits!");
    exit;
}

// ===== 4. HELPER FUNCTIONS =====
function sendPreCheckoutResponse($bot_token, $query_id, $ok, $error_message = '') {
    $url = "https://api.telegram.org/bot$bot_token/answerPreCheckoutQuery";
    $data = [
        'pre_checkout_query_id' => $query_id,
        'ok' => $ok
    ];
    if (!$ok) {
        $data['error_message'] = $error_message;
    }

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    $response = curl_exec($ch);
    error_log("AnswerPreCheckoutQuery Response: " . $response);
    curl_close($ch);
}

function sendMessage($bot_token, $chat_id, $text) {
    $url = "https://api.telegram.org/bot$bot_token/sendMessage";
    $data = [
        'chat_id' => $chat_id,
        'text' => $text
    ];

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    $response = curl_exec($ch);
    error_log("SendMessage Response: " . $response);
    curl_close($ch);
}