<?php  
require 'config.php';  
header('Content-Type: application/json');  
  
function log_error($message) {  
    error_log("[Draw API] " . $message);  
}  
  
function sendTelegramMessage($chat_id, $text) {  
    global $bot_token; // from config.php  
    $url = "https://api.telegram.org/bot$bot_token/sendMessage";  
    $data = [  
        'chat_id' => $chat_id,  
        'text' => $text,  
        'parse_mode' => 'HTML'  
    ];  
  
    $ch = curl_init($url);  
    curl_setopt($ch, CURLOPT_POST, 1);  
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);  
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);  
    $result = curl_exec($ch);  
    if (curl_errno($ch)) {  
        log_error("Telegram send error: " . curl_error($ch));  
    }  
    curl_close($ch);  
    return $result;  
}  
  
try {  
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);  
  
    $user_id  = $_GET['user_id'] ?? null;  
    $action   = $_GET['action'] ?? null;  
    $tickets  = isset($_GET['tickets']) ? intval($_GET['tickets']) : 0;  
    $username = $_GET['username'] ?? null;  
  
    if (!$user_id) {  
        echo json_encode(['ok' => false, 'message' => 'User ID is required']);  
        exit;  
    }  
  
    // ✅ Check if user is registered before doing anything  
    $checkUser = $pdo->prepare("SELECT ticket_point FROM users WHERE user_id = ?");  
    $checkUser->execute([$user_id]);  
    $registeredUser = $checkUser->fetch(PDO::FETCH_ASSOC);  
  
    if (!$registeredUser) {  
        echo json_encode(['ok' => false, 'message' => 'You must register before participating in the draw.']);  
        exit;  
    }  
  
    log_error("Received: user_id=$user_id, action=$action, tickets=$tickets, username=" . var_export($username, true));  
  
    // ====== DETAILS ======  
    if ($action === 'details') {  
        try {  
            // Total tickets  
            $stmt = $pdo->query("SELECT COALESCE(SUM(tickets),0) FROM draw_entries");  
            $total = $stmt->fetchColumn();  
  
            // User tickets  
            $stmt = $pdo->prepare("SELECT COALESCE(SUM(tickets),0) FROM draw_entries WHERE user_id = ?");  
            $stmt->execute([$user_id]);  
            $user_tickets = $stmt->fetchColumn();  
  
            // User points  
            $user_points = (int)$registeredUser['ticket_point'];  
  
            $winning_chance = $total > 0 ? round(($user_tickets / $total) * 100, 2) . "%" : "0.00%";  
  
            // Last 5 winners  
            $stmt = $pdo->query("SELECT winner_name FROM draws WHERE winner_name IS NOT NULL ORDER BY id DESC LIMIT 5");  
            $winners = $stmt->fetchAll(PDO::FETCH_COLUMN);  
            $winners_response = $winners ?: ["No recent winners"];  
  
            echo json_encode([  
                'ok' => true,  
                'user_tickets' => (int)$user_tickets,  
                'user_points' => $user_points,  
                'total_tickets' => (int)$total,  
                'winning_chance' => $winning_chance,  
                'last_winners' => $winners_response,  
                'prize_per_winner' => 100000  
            ]);  
        } catch (Exception $e) {  
            log_error("DETAILS error: " . $e->getMessage());  
            echo json_encode(['ok' => false, 'message' => 'Failed to load details']);  
        }  
        exit;  
    }  
  
    // ====== PARTICIPATE ======  
    if ($action === 'participate') {  
        if ($tickets <= 0 || $tickets > 20) {  
            echo json_encode(['ok' => false, 'message' => 'Ticket amount must be between 1 and 20']);  
            exit;  
        }  
  
        try {  
            $pdo->beginTransaction();  
  
            // Calculate required points (1 ticket = 30 points)  
            $required_points = $tickets * 30;  
  
            // Ensure user has enough points  
            if ($registeredUser['ticket_point'] < $required_points) {  
                $pdo->rollBack();  
                echo json_encode(['ok' => false, 'message' => "Not enough points. Required: $required_points, Available: {$registeredUser['ticket_point']}"]);  
                exit;  
            }  
  
            // Check user's current tickets  
            $stmt = $pdo->prepare("SELECT COALESCE(SUM(tickets),0) FROM draw_entries WHERE user_id = ?");  
            $stmt->execute([$user_id]);  
            $current_user_tickets = $stmt->fetchColumn();  
  
            if ($current_user_tickets + $tickets > 50) {  
                $pdo->rollBack();  
                echo json_encode(['ok' => false, 'message' => "Maximum 50 tickets per user. You already have $current_user_tickets tickets."]);  
                exit;  
            }  
  
            // Deduct points  
            $stmt = $pdo->prepare("UPDATE users SET ticket_point = ticket_point - ? WHERE user_id = ?");  
            $stmt->execute([$required_points, $user_id]);  
            log_error("Deducted $required_points points ($tickets tickets) for user_id=$user_id");  
  
            // Use provided username or fallback  
            $final_username = !empty($username) ? $username : (string)$user_id;  
  
            // Insert entry  
            $stmt = $pdo->prepare("INSERT INTO draw_entries (user_id, username, tickets) VALUES (?, ?, ?)");  
            $stmt->execute([$user_id, $final_username, $tickets]);  
  
            // Check if draw reached 20000 tickets  
            $stmt = $pdo->query("SELECT COALESCE(SUM(tickets),0) FROM draw_entries");  
            $total = $stmt->fetchColumn();  
  
            if ($total >= 20000) {  
                // Lock and get all entries  
                $stmt = $pdo->query("SELECT user_id, username FROM draw_entries FOR UPDATE");  
                $entries = $stmt->fetchAll(PDO::FETCH_ASSOC);  
  
                if ($entries) {  
                    $winners = [];  
                    $reward = 100000;  
  
                    for ($i = 0; $i < 5; $i++) {  
                        if (empty($entries)) break;  
                        $winnerIndex = array_rand($entries);  
                        $winner = $entries[$winnerIndex];  
                        unset($entries[$winnerIndex]);  
  
                        // Update winner balance  
                        $stmt = $pdo->prepare("UPDATE users SET balance = balance + ? WHERE user_id = ?");  
                        $stmt->execute([$reward, $winner['user_id']]);  
  
                        // Log winner  
                        $stmt = $pdo->prepare("INSERT INTO draws (user_id, winner_name, reward) VALUES (?, ?, ?)");  
                        $stmt->execute([$winner['user_id'], $winner['username'], $reward]);  
  
                        // Notify  
                        sendTelegramMessage($winner['user_id'], "🎉 Congratulations {$winner['username']}! You won 100k MEGA in today’s draw!");  
  
                        $winners[] = $winner;  
                    }  
  
                    // Reset entries  
                    $pdo->query("TRUNCATE TABLE draw_entries");  
  
                    $pdo->commit();  
                    echo json_encode([  
                        'ok' => true,  
                        'message' => 'Draw completed',  
                        'winners' => $winners,  
                        'prize_per_winner' => $reward  
                    ]);  
                    exit;  
                }  
            }  
  
            $pdo->commit();  
            echo json_encode([  
                'ok' => true,  
                'message' => 'Participation successful',  
                'data' => [  
                    'tickets_entered' => $tickets,  
                    'points_deducted' => $required_points,  
                    'new_point_balance' => $registeredUser['ticket_point'] - $required_points  
                ]  
            ]);  
        } catch (Exception $e) {  
            $pdo->rollBack();  
            log_error("PARTICIPATE error: " . $e->getMessage());  
            echo json_encode(['ok' => false, 'message' => 'Failed to participate: ' . $e->getMessage()]);  
        }  
        exit;  
    }  
  
    echo json_encode(['ok' => false, 'message' => 'Invalid action']);  
  
} catch (Exception $e) {  
    log_error("GENERAL error: " . $e->getMessage());  
    echo json_encode(['ok' => false, 'message' => 'Server error']);  
}  
?>