<?php
// Include dependencies
require 'config.php';

header('Content-Type: application/json');

// Ensure PDO is defined
if (!isset($pdo) || !($pdo instanceof PDO)) {
    echo json_encode(['ok' => false, 'message' => 'Database connection not initialized.']);
    exit;
}

// Get user_id from URL parameter
$user_id = isset($_GET['user_id']) ? $_GET['user_id'] : null;
if (empty($user_id)) {
    echo json_encode(['ok' => false, 'message' => 'Missing user_id parameter.']);
    exit;
}

error_log("key-webhook.php: Received callback for user_id=$user_id");

// Read and decode input JSON
$input = file_get_contents("php://input");
if (empty($input)) {
    error_log("key-webhook.php: No input data for user_id=$user_id");
    echo json_encode(['ok' => false, 'message' => 'No input data provided.']);
    exit;
}
$data = json_decode($input, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    error_log("key-webhook.php: Invalid JSON input for user_id=$user_id: " . json_last_error_msg());
    echo json_encode(['ok' => false, 'message' => 'Invalid JSON input: ' . json_last_error_msg()]);
    exit;
}
error_log("key-webhook.php: Input data for user_id=$user_id: " . print_r($data, true));

// Validate required fields
if (!isset($data['status']) || !isset($data['txs']) || !is_array($data['txs']) || empty($data['txs']) || !isset($data['txs'][0]['received_amount'])) {
    error_log("key-webhook.php: Invalid data for user_id=$user_id: " . print_r($data, true));
    echo json_encode(['ok' => false, 'message' => 'Missing or invalid status or txs data.']);
    exit;
}

// Check if status is Paid
if ($data['status'] !== 'Paid') {
    error_log("key-webhook.php: Invalid status for user_id=$user_id: " . $data['status']);
    echo json_encode(['ok' => false, 'message' => 'Transaction status is not Paid.']);
    exit;
}

// Fetch user
try {
    $stmt = $pdo->prepare("SELECT id, user_id FROM users WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        error_log("key-webhook.php: User not found for user_id=$user_id");
        echo json_encode(['ok' => false, 'message' => 'User not found']);
        exit;
    }
} catch (PDOException $e) {
    error_log("key-webhook.php: Database error fetching user for user_id=$user_id: " . $e->getMessage());
    echo json_encode(['ok' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    exit;
}

// Calculate keys to add based on exchange rates
$received_amount = (float)$data['txs'][0]['received_amount'];

// Base rate: 1 key = 0.02 USDT (5 keys = 0.1 USDT, 50 keys = 1 USDT)
$keys_to_add = floor($received_amount / 0.02);

// Adjust for bulk rate (e.g., 1000 keys = 30 USDT, 1 key = 0.03 USDT) if applicable
if ($received_amount >= 30) {
    $bulk_keys = floor($received_amount / 0.03);
    $keys_to_add = max($keys_to_add, $bulk_keys); // Use the higher key count
}

if ($keys_to_add <= 0) {
    error_log("key-webhook.php: Invalid keys to add ($keys_to_add) for user_id=$user_id, received_amount=$received_amount");
    echo json_encode(['ok' => false, 'message' => 'Invalid deposit amount, no keys added.']);
    exit;
}

// Update users.key
try {
    $pdo->beginTransaction();
    $stmt = $pdo->prepare("UPDATE users SET `key` = `key` + ? WHERE user_id = ?");
    $stmt->execute([$keys_to_add, $user_id]);
    if ($stmt->rowCount() === 0) {
        $pdo->rollBack();
        error_log("key-webhook.php: No rows updated for user_id=$user_id");
        echo json_encode(['ok' => false, 'message' => 'No user found to update keys.']);
        exit;
    }
    $pdo->commit();
    error_log("key-webhook.php: Successfully added $keys_to_add keys for user_id=$user_id");
} catch (PDOException $e) {
    $pdo->rollBack();
    error_log("key-webhook.php: Database error updating keys for user_id=$user_id: " . $e->getMessage());
    echo json_encode(['ok' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    exit;
}

// Final response
echo json_encode([
    'ok' => true,
    'message' => 'Keys added successfully',
    'keys_added' => $keys_to_add
], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
?>