<?php
require 'config.php';

// ===== LOGGING SETUP =====
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/buy_keys.log');

// ===== 1. READ INPUT =====
$user_id = $_GET['user_id'] ?? null;
$input = file_get_contents('php://input');
$json_data = json_decode($input, true) ?? [];
$keys = (int) ($json_data['amount'] ?? $json_data['keys'] ?? 0);

error_log("=== Buy Keys Request at " . date('Y-m-d H:i:s') . " ===");
error_log("Raw Input: " . $input);
error_log("Parsed => user_id: " . ($user_id ?? 'NULL') . ", keys: " . $keys);

// ===== 2. VALIDATE INPUT =====
if (!$user_id || $keys <= 0 || $keys % 5 !== 0) {
    $error = "Invalid request (user_id: " . ($user_id ?? 'missing') . ", keys: $keys)";
    error_log("ERROR: $error");
    echo json_encode(['ok' => false, 'message' => $error]);
    exit;
}

// ===== 3. CALCULATE PRICE =====
// 3.6 Stars per key, no *100 because Stars use integer units
$total_price = (int) floor($keys * 3.6);
error_log("Calculated total price: $total_price XTR for $keys keys");

// ===== 4. PREPARE PAYLOAD =====
$payload = $user_id . '|' . $keys . '|' . time();

$payload_data = [
    'title' => "Buy $keys Keys",
    'description' => "Purchase $keys keys for MegaTask",
    'payload' => $payload,
    'currency' => 'XTR',
    'prices' => [
        ['label' => "$keys Keys", 'amount' => $total_price]
    ]
];

// ===== 5. SEND TO TELEGRAM API =====
$telegram_url = "https://api.telegram.org/bot{$config['bot_token']}/createInvoiceLink";
$ch = curl_init($telegram_url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload_data));
$response = curl_exec($ch);

if ($response === false) {
    $curl_error = curl_error($ch);
    error_log("cURL ERROR: $curl_error");
    echo json_encode(['ok' => false, 'message' => 'Telegram API request failed: ' . $curl_error]);
    curl_close($ch);
    exit;
}
curl_close($ch);

// ===== 6. PARSE TELEGRAM RESPONSE =====
error_log("Telegram API Response: " . $response);
$decoded = json_decode($response, true);

if (!empty($decoded['ok']) && isset($decoded['result'])) {
    // SUCCESS: Return invoice link
    echo json_encode([
        'ok' => true,
        'invoice_link' => $decoded['result'],
        'star_cost' => $total_price
    ]);
} else {
    $error_message = $decoded['description'] ?? 'Unknown Telegram API error';
    $error_code = $decoded['error_code'] ?? 0;
    error_log("Telegram API Error: $error_code - $error_message");
    echo json_encode(['ok' => false, 'message' => "$error_code: $error_message"]);
    exit;
}

// ===== 7. (OPTIONAL) AUTO-SET WEBHOOK =====
// If you want to make sure webhook is always correct, uncomment this block:
$webhook_url = "https://wenepay.cyou/scratch/api/webhook.php";
$setWebhookUrl = "https://api.telegram.org/bot{$config['bot_token']}/setWebhook?url=" . urlencode($webhook_url);
$webhook_response = file_get_contents($setWebhookUrl);
error_log("Webhook Set Response: " . $webhook_response);