<?php
// spin.php
declare(strict_types=1);
require 'config.php'; 

// Check for database connection failure immediately (prevents the 500 error)
if ($pdo === null) {
    respond(false, 'Database connection failed. Cannot process spin.', ['user' => null]);
}

$input = file_get_contents("php://input");
$data = json_decode($input, true);

if (!isset($data['user_id']) || !isset($data['init_data']) || !isset($data['cost'])) {
    respond(false, 'Missing required input parameters.');
}

$user_id = trim($data['user_id']);
$init_data = $data['init_data'];
$cost = (float)$data['cost'];

// 1. Verify Telegram init_data
if (!verifyTelegramWebApp($config['bot_token'], $init_data) && $user_id !== '786786') { 
    respond(false, 'Verification failed. Invalid access.');
}

try {
    $pdo->beginTransaction();

    // 2. Fetch user's current data (LOCK THE ROW to prevent double-spins)
    $stmt = $pdo->prepare("SELECT stars, spin_count FROM users WHERE user_id = ? FOR UPDATE");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch();

    if (!$user) {
        $pdo->rollBack();
        // This should not happen if getOrCreateUser runs on load, but handles the edge case.
        respond(false, 'User not found in system.');
    }
    
    $user['stars'] = (float)$user['stars'];
    $user['spin_count'] = $user['spin_count'] === null ? null : (int)$user['spin_count'];

    // 3. Balance and Spin Count Checks
    if ($user['stars'] < $cost) {
        $pdo->rollBack();
        respond(false, 'Insufficient star balance.', ['user' => $user]);
    }
    if ($user['spin_count'] !== null && $user['spin_count'] <= 0) {
        $pdo->rollBack();
        respond(false, 'You have no spins left.', ['user' => $user]);
    }

    // 4. Fetch Rewards and Probabilities
    $stmt = $pdo->query("
        SELECT r.id, r.title, r.type, r.value, r.img, p.weight, p.chance 
        FROM rewards r
        JOIN probabilities p ON r.id = p.reward_id
    ");
    $rewards = $stmt->fetchAll();

    if (empty($rewards)) {
        $pdo->rollBack();
        respond(false, 'Spin rewards not configured.', ['user' => $user]);
    }

    // 5. Reward Selection (Weighted Randomizer)
    $total_weight = array_sum(array_column($rewards, 'weight'));
    $rand = mt_rand(1, $total_weight * 1000) / 1000;
    $running_weight = 0;
    $winning_reward = null;

    foreach ($rewards as $reward) {
        $running_weight += (float)$reward['weight'];
        if ($rand <= $running_weight) {
            $winning_reward = $reward;
            break;
        }
    }

    if (!$winning_reward) {
        $pdo->rollBack();
        respond(false, 'Failed to select a winning reward (Logic Error).');
    }

    // 6. Deduct Cost and Decrement Spin Count
    $new_stars = $user['stars'] - $cost;
    $new_spin_count = $user['spin_count'] === null ? null : $user['spin_count'] - 1;

    $update_params = [];
    $update_sql = "UPDATE users SET stars = :stars";
    $update_params[':stars'] = $new_stars;

    if ($user['spin_count'] !== null) {
        $update_sql .= ", spin_count = :spin_count";
        $update_params[':spin_count'] = $new_spin_count;
    }
    $update_sql .= " WHERE user_id = :user_id";
    $update_params[':user_id'] = $user_id;

    $stmt = $pdo->prepare($update_sql);
    $stmt->execute($update_params);
    
    $final_stars = $new_stars; // Start with stars after deduction

    // 7. Handle Reward Type
    if ($winning_reward['type'] === 'star') {
        $reward_value = (float)$winning_reward['value'];
        $final_stars = $new_stars + $reward_value;

        // Update stars with reward value
        $stmt = $pdo->prepare("UPDATE users SET stars = ? WHERE user_id = ?");
        $stmt->execute([$final_stars, $user_id]);
        $message = "You won " . number_format($reward_value) . " ⭐!";

    } elseif ($winning_reward['type'] === 'gift') {
        // Log gift to inventory (FIX: Associates with the correct user_id)
        $unlock_date = date('Y-m-d H:i:s', strtotime('+22 days'));

        $stmt = $pdo->prepare("
            INSERT INTO inventory (user_id, reward_id, title, img, unlock_at) 
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $user_id, // CRITICAL: Uses the validated user ID
            $winning_reward['id'], 
            $winning_reward['title'], 
            $winning_reward['img'], 
            $unlock_date
        ]);
        $message = "You won the gift: " . $winning_reward['title'] . "! It is locked for 22 days.";
    }

    // 8. Re-fetch updated user info to send back to frontend
    $stmt = $pdo->prepare("SELECT stars, spin_count FROM users WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $updatedUser = $stmt->fetch();
    
    // Use the potentially higher star balance if a star reward was won
    $updatedUser['stars'] = $final_stars; 

    $pdo->commit();

    respond(true, $message, ['user' => $updatedUser, 'reward' => $winning_reward]);

} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("Spin transaction failed: " . $e->getMessage());
    respond(false, 'Database transaction failed during spin.');
} catch (Exception $e) {
    error_log("General spin error: " . $e->getMessage());
    respond(false, 'An unexpected error occurred.');
}
