<?php
// get_user_info.php - CORRECTED VERSION (No change needed for difficulty)
declare(strict_types=1);

require 'config.php';
require 'init_data_check.php'; // Assuming this file contains verifyTelegramWebApp()
header('Content-Type: application/json');

// Helper function for consistent JSON responses
function respond(bool $ok, string $message, array $extra = []): void {
    $response = ['ok' => $ok, 'message' => $message];
    if (!empty($extra)) $response = array_merge($response, $extra); // Merge all extra data (user AND inventory)
    echo json_encode($response);
    exit;
}

// Function to fetch Telegram Star Gifts and their prices
function fetchStarGifts(string $botToken): array {
    $url = "https://api.telegram.org/bot{$botToken}/getStarGifts?hash=0";  // Added hash=0
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPGET, true);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    error_log("StarGifts raw response: " . $response);  // Log raw response for debugging
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        error_log("StarGifts decoded data: " . json_encode($data));  // Log decoded data
        
        if ($data['ok']) {
            // Return map of gift ID to stars (price)
            $gifts = [];
            foreach ($data['result']['gifts'] ?? [] as $gift) {
                $gifts[$gift['id']] = $gift['stars'] ?? 0;  // Correct field: 'stars'
            }
            return $gifts;
        }
    }
    // On error, return empty
    error_log("Failed to fetch Star Gifts: HTTP {$httpCode} - " . ($response ?? 'No response'));
    return [];
}

// Read input JSON
$input = file_get_contents("php://input");
$data = json_decode($input, true);

if (!isset($data['user_id']) || !isset($data['init_data'])) {
    respond(false, 'Missing user_id or init_data.');
}

$user_id = trim($data['user_id']);
$init_data = $data['init_data'];
$inventory = []; // Initialize inventory array

// 1. Verify Telegram init_data
if ($init_data !== '786786') { // Allow bypass for local testing if needed
    $verification = verifyTelegramWebApp($config['bot_token'], $init_data);
    if (!$verification['ok']) {
        respond(false, 'Init data verification failed.');
    }
    $init_data_array = $verification['data'];
    if (!isset($init_data_array['user']['id']) || (string)$init_data_array['user']['id'] !== $user_id) {
        respond(false, 'User ID mismatch.');
    }
}

// Fetch Star Gifts prices once (assuming reward_id matches Telegram gift ID)
$giftPrices = fetchStarGifts($config['bot_token']);

// Simple title-to-price map if IDs don't match (adjust based on your logs/API response)
$titleToPriceMap = [  // Example mappings; replace with actual from logs
    'Cookiehart' => 100,  // Placeholder; get real from API
    'Lollipop' => 150,
    'Desk Calendar' => 200,
    'Coffee Cup' => 250,
    'Champagne' => 500,
    'Rose' => 300,
    // Add more as needed
];

try {
    // 2. Fetch user data (stars, balance, and spin_count)
    $stmt = $pdo->prepare("SELECT stars, balance, spin_count FROM users WHERE user_id = ?"); 
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        // User registration logic if user doesn't exist
        // Setting Stars high, and Mega (balance) low/zero to make it rare
        $default_stars = 1000;
        $default_balance = 0; // Mega balance starts at 0
        $stmt = $pdo->prepare("INSERT INTO users (user_id, stars, balance, spin_count) VALUES (?, ?, ?, ?)");
        $stmt->execute([$user_id, $default_stars, $default_balance, null]); 
        
        $user = ['stars' => $default_stars, 'balance' => $default_balance, 'spin_count' => null];
        $message = 'User registered and initial balance fetched.';
    } else {
        $message = 'User info fetched successfully.';
    }
    
    // ⭐ 3. Fetch user's inventory (gifts) ⭐
    $stmt = $pdo->prepare("
        SELECT reward_id, title, img, unlock_at
        FROM inventory
        WHERE user_id = ?
        ORDER BY unlock_at DESC
    ");
    $stmt->execute([$user_id]);
    $inventory = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Add price to each inventory item (try ID first, fallback to title map)
    foreach ($inventory as &$item) {
        $giftId = $item['reward_id'];
        $item['price'] = $giftPrices[$giftId] ?? $titleToPriceMap[$item['title']] ?? 0;  // Fallback to title map
    }
    unset($item);  // Unset reference

    // 4. Return user data AND inventory
    respond(true, $message, [
        'user' => [
            'stars' => (float)$user['stars'],
            'balance' => (float)($user['balance'] ?? 0), // 'balance' is the Mega currency
            'spin_count' => $user['spin_count']
        ],
        // The frontend JS looks for this key!
        'inventory' => $inventory 
    ]);

} catch (PDOException $e) {
    error_log("Database error in get_user_info: " . $e->getMessage());
    respond(false, 'Database error.');
}
?>