<?php
declare(strict_types=1); // Enable strict typing for better error catching

require 'config.php';
require 'init_data_check.php';
header('Content-Type: application/json');

// Ensure PDO connection
if (!isset($pdo) || !($pdo instanceof PDO)) {
    respond(false, 'Database not initialized.');
}

// Read input JSON
$input = file_get_contents("php://input");
if (empty($input)) {
    respond(false, 'No input data provided.');
}
$data = json_decode($input, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    respond(false, 'Invalid JSON input: ' . json_last_error_msg());
}

if (!isset($data['user_id']) || !isset($data['init_data'])) {
    respond(false, 'Missing user_id or init_data.');
}

$user_id = trim($data['user_id']);
$init_data = $data['init_data'];

// Validate Telegram ID format (only numbers, at least 5 digits)
if (!ctype_digit($user_id) || strlen($user_id) < 5) {
    respond(false, 'Invalid user ID format.');
}

// Verify Telegram init_data unless bypass key is used
if ($init_data !== '786786') { // Consider making this configurable in config.php
    $verification = verifyTelegramWebApp($config['bot_token'], $init_data);
    if (!$verification['ok']) {
        respond(false, 'Init data verification failed: ' . ($verification['message'] ?? 'Unknown error'));
    }
    $init_data_array = $verification['data'];
    if (!isset($init_data_array['user']['id']) || (string)$init_data_array['user']['id'] !== $user_id) {
        respond(false, 'User ID mismatch.');
    }
}

try {
    // Check registration
    $stmt = $pdo->prepare("SELECT id, user_id, name, balance, ticket_point, scratch_card, `key` FROM users WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        respond(false, 'You must register before spinning.');
    }

    // Check for enough keys
    if ((int)$user['key'] < 1) {
        respond(false, 'Insufficient keys. You need at least 1 key to spin.');
    }

    // Define rewards with probabilities (out of 100,000 for fine-grained control)
    $reward_options = [
        ['type' => 'MEGA', 'value' => 500, 'description' => '500 MEGA', 'weight' => 200],
        ['type' => 'MEGA', 'value' => 100, 'description' => '100 MEGA', 'weight' => 200], // Cumulative: 400
        ['type' => 'USDT', 'value' => 0.05, 'description' => '0.05 USDT', 'weight' => 200], // Cumulative: 600
        ['type' => 'TICKET', 'value' => 50, 'description' => '1 Ticket', 'weight' => 200], // Cumulative: 800
        ['type' => 'USDT', 'value' => 0.1, 'description' => '0.1 USDT', 'weight' => 1200], // Cumulative: 2000
        ['type' => 'KEY', 'value' => 1, 'description' => '1 Key', 'weight' => 8000], // Cumulative: 10000
        ['type' => 'DIAMOND_CARD', 'value' => 1, 'description' => '1 Diamond Card', 'weight' => 8000], // Cumulative: 18000
        ['type' => 'NFT', 'value' => 1, 'description' => 'NFT', 'weight' => 0000], // 1% chance, Cumulative: 19000
        ['type' => 'MEGA', 'value' => 30, 'description' => '30 MEGA', 'weight' => 81000], // Fills the rest to 100000
    ];

    // Random spin: Select reward based on weighted random
    $roll = mt_rand(1, 100000);
    $cumulative = 0;
    $reward = null;
    foreach ($reward_options as $option) {
        $cumulative += $option['weight'];
        if ($roll <= $cumulative) {
            $reward = $option;
            break;
        }
    }

    // Process spin in transaction
    $pdo->beginTransaction();

    // Deduct 1 key
    $stmt = $pdo->prepare("UPDATE users SET `key` = `key` - 1 WHERE user_id = ?");
    $stmt->execute([$user_id]);

    // Apply reward
    switch ($reward['type']) {
        case 'MEGA':
            $stmt = $pdo->prepare("UPDATE users SET balance = balance + ? WHERE user_id = ?");
            $stmt->execute([$reward['value'], $user_id]);
            break;

        case 'USDT':
            $stmt = $pdo->prepare("SELECT id FROM mytask WHERE id = ?");
            $stmt->execute([$user['id']]);
            if ($stmt->fetch()) {
                $stmt = $pdo->prepare("UPDATE mytask SET task_balance = task_balance + ? WHERE id = ?");
                $stmt->execute([$reward['value'], $user['id']]);
            } else {
                $stmt = $pdo->prepare("INSERT INTO mytask (id, task_balance) VALUES (?, ?)");
                $stmt->execute([$user['id'], $reward['value']]);
            }
            break;

        case 'TICKET':
            $stmt = $pdo->prepare("UPDATE users SET ticket_point = ticket_point + ? WHERE user_id = ?");
            $stmt->execute([$reward['value'], $user_id]);
            break;

        case 'KEY':
            $stmt = $pdo->prepare("UPDATE users SET `key` = `key` + ? WHERE user_id = ?");
            $stmt->execute([$reward['value'], $user_id]);
            break;

        case 'DIAMOND_CARD':
            $scratch_card = json_decode($user['scratch_card'] ?? '[]', true);
            if (!is_array($scratch_card)) {
                $scratch_card = [];
            }
            $card_id = bin2hex(random_bytes(6));
            $scratch_card[] = ['card_id' => $card_id, 'card_type' => 'diamond'];
            $stmt = $pdo->prepare("UPDATE users SET scratch_card = ? WHERE user_id = ?");
            $stmt->execute([json_encode($scratch_card, JSON_UNESCAPED_SLASHES), $user_id]);
            break;

        case 'NFT':
            // Randomly select Rose or Bear
            $nft_types = ['Rose', 'Bear'];
            $nft_type = $nft_types[array_rand($nft_types)];
            $nft_id = bin2hex(random_bytes(6)); // Unique NFT ID
            $created_at = date('Y-m-d H:i:s');

            // Create NFT table if not exists
            $pdo->exec("CREATE TABLE IF NOT EXISTS nft_wins (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id BIGINT NOT NULL,
                nft_id VARCHAR(12) NOT NULL,
                nft_type VARCHAR(50) NOT NULL,
                created_at DATETIME NOT NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");

            // Insert NFT win
            $stmt = $pdo->prepare("INSERT INTO nft_wins (user_id, nft_id, nft_type, created_at) VALUES (?, ?, ?, ?)");
            $stmt->execute([$user_id, $nft_id, $nft_type, $created_at]);

            // Update reward description for response
            $reward['description'] = $nft_type . ' NFT';
            break;
    }

    $pdo->commit();

    // Optional: Log the spin for auditing (uncomment and adjust path if needed)
    // error_log("Spin by user {$user_id}: Reward - {$reward['description']}\n", 3, '/var/log/spin.log');

    respond(true, 'Spin completed successfully. 1 key deducted.', $reward['description']);
} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    // Log error internally without exposing to user
    error_log("Database error in spin: " . $e->getMessage());
    respond(false, 'An unexpected error occurred. Please try again.');
} catch (Exception $e) {
    respond(false, 'Unexpected error: ' . $e->getMessage());
}

// Helper function for consistent JSON responses
function respond(bool $ok, string $message, ?string $reward_desc = null): void {
    $response = ['ok' => $ok, 'message' => $message];
    if ($reward_desc) {
        $response['reward'] = $reward_desc;
    }
    echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}