<?php
/**
 * deposit.php
 * Clean & fixed version
 */

header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('Asia/Kolkata');

// 🧪 Show error in log (but not to user)
ini_set('display_errors', 0);
ini_set('log_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/init_data_validator.php';

/**
 * ✅ Success response
 */
function json_ok($payload = []) {
    http_response_code(200);
    echo json_encode(array_merge(['ok' => true], $payload), JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * ❌ Fail response
 */
function json_fail($message, $code = 400) {
    http_response_code($code);
    echo json_encode(['ok' => false, 'message' => $message], JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * 🔁 Get full callback URL
 */
function getDepositCallbackUrl($user_id) {
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? '';
    $directory = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/') . '/';
    return $protocol . '://' . $host . $directory . 'deposit-callback.php?user_id=' . urlencode($user_id);
}

/**
 * 📨 Read & parse JSON body
 */
$body = file_get_contents('php://input');
$input = json_decode($body, true);

if (!is_array($input) || empty($input['init_data']) || !isset($input['amount'])) {
    json_fail('Invalid request. Expecting init_data and amount.');
}

$init_data = $input['init_data'];
$amount = (float) $input['amount'];

if ($amount <= 0) {
    json_fail('Invalid amount.');
}

// 🛢 Load settings from DB
try {
    $stmt = $pdo->query("SELECT merchant_api_key, bot_token, min_deposit, max_deposit FROM settings ORDER BY id DESC LIMIT 1");
    $settings = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$settings) {
        json_fail('Settings not found.', 500);
    }

    $bot_token = $settings['bot_token'];
    $merchant_api_key = $settings['merchant_api_key'];
    $min_deposit = (float) ($settings['min_deposit'] ?? 0);
    $max_deposit = (float) ($settings['max_deposit'] ?? 0);
} catch (PDOException $e) {
    json_fail('DB error fetching settings: ' . $e->getMessage(), 500);
}

// 🔐 Verify Telegram init_data
$verify = verifyTelegramWebApp($bot_token, $init_data);
if (empty($verify['ok']) || empty($verify['data']['user']['id'])) {
    json_fail('Invalid init_data signature (verification failed).', 401);
}

$telegram_id = (int) $verify['data']['user']['id'];

// 👤 Get internal user_id
try {
    $stmt = $pdo->prepare("SELECT user_id FROM users WHERE telegram_id = ? LIMIT 1");
    $stmt->execute([$telegram_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        json_fail('User not registered. Please register first.', 404);
    }
    $internal_user_id = $user['user_id'];
} catch (PDOException $e) {
    json_fail('DB error fetching user: ' . $e->getMessage(), 500);
}

// 💰 Amount validation
if ($amount < $min_deposit) {
    json_fail("Amount is below minimum deposit ({$min_deposit}).");
}
if ($max_deposit > 0 && $amount > $max_deposit) {
    json_fail("Amount exceeds maximum deposit ({$max_deposit}).");
}

// 🧾 OxaPay API call
try {
    $url = "https://api.oxapay.com/v1/payment/white-label";
    $callback_url = getDepositCallbackUrl($internal_user_id);

    $payload = [
        "pay_currency" => "USDT",
        "amount"       => $amount,
        "currency"     => "USDT",
        "network"      => "BEP20",
        "lifetime"     => 15,
        "callback_url" => $callback_url
    ];

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Content-Type: application/json",
        "merchant_api_key: {$merchant_api_key}"
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);

    $resp = curl_exec($ch);
    if ($resp === false) {
        $err = curl_error($ch);
        curl_close($ch);
        json_fail("Curl error: " . $err, 500);
    }

    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $json = json_decode($resp, true);
    if ($http_code !== 200 || empty($json['status']) || $json['status'] != 200 || empty($json['data'])) {
        json_fail("Failed to generate white-label invoice. (Code: $http_code)", 500);
    }

    $data = $json['data'];
    if (empty($data['track_id']) || empty($data['address']) || empty($data['pay_amount'])) {
        json_fail('Unexpected response from payment provider: missing fields.', 500);
    }

    // ✅ Final success response
    json_ok([
        'track_id' => (string) $data['track_id'],
        'address'  => (string) $data['address'],
        'amount'   => (string) $data['pay_amount']
    ]);

} catch (Exception $e) {
    json_fail('White-label invoice generation failed: ' . $e->getMessage(), 500);
}