<?php
// Ensure this file exists and contains your database connection ($pdo)
require_once 'config.php'; 

header('Content-Type: application/json');

// Read JSON input (user_id is expected)
$input = json_decode(file_get_contents('php://input'), true);
$user_id = $input['user_id'] ?? '';
$action = $input['action'] ?? 'claim';

// Default response structure
$response = ['ok' => false, 'message' => '', 'redirect' => '', 'result' => []];

try {
    if (empty($user_id)) {
        $response['message'] = 'User ID is required.';
        echo json_encode($response);
        exit;
    }

    // --- 1. Fetch User Data ---
    // Make sure your 'users' table has 'balance', 'last_claim', and 'consecutive_days' columns.
    $stmt = $pdo->prepare("SELECT balance, last_claim, consecutive_days FROM users WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        $response['message'] = 'User not found.';
        echo json_encode($response);
        exit;
    }

    // Initialize variables
    $today = new DateTime(date('Y-m-d'));
    $last_claim_str = $user['last_claim'] ?? null;
    $current_streak = (int)$user['consecutive_days'];

    if ($action === 'get_status') {
        $response['ok'] = true;
        $response['result'] = [
            'current_streak' => $current_streak,
            'last_claim' => $last_claim_str,
            'balance' => $user['balance']
        ];
        echo json_encode($response);
        exit;
    }

    // Proceed with claim if action is 'claim'
    $mega_reward = 0;
    $message = '';

    // --- 2. Check Daily Claim Status ---
    if ($last_claim_str === $today->format('Y-m-d')) {
        $response['message'] = 'You already claimed today. Try again tomorrow!';
        $response['result'] = [
            'current_streak' => $current_streak,
            'last_claim' => $last_claim_str
        ];
        echo json_encode($response);
        exit;
    }

    // --- 3. Calculate New Streak ---
    $new_streak = 1; // Default if this is the first claim or a missed day
    
    if ($last_claim_str) {
        $last_claim = new DateTime($last_claim_str);
        $interval = $today->diff($last_claim);
        $days_diff = (int)$interval->format('%a');

        if ($days_diff === 1) {
            // Consecutive day claimed!
            $new_streak = $current_streak + 1;
        } else if ($days_diff > 1) {
            // Missed a day, streak is reset to 1
            $message .= "Streak broken! Resetting to day 1. ";
        } 
        // If $days_diff == 0, the first check would have caught it.
    }
    
    // --- 4. Determine Rewards based on Streak ---
    $message .= "Claiming Day **$new_streak** reward. ";

    switch ($new_streak) {
        case 5:
            $mega_reward = 50;
            $message .= "✅ **5-Day Bonus!** ";
            break;
        case 10:
            $mega_reward = 150;
            $message .= "⭐️ **10-Day Bonus!** ";
            break;
        case 15:
            $mega_reward = 300;
            $message .= "🔥 **15-Day Mega Bonus!** ";
            break;
        case 30:
            $mega_reward = 600;
            $message .= "👑 **30-Day GRAND PRIZE!** ";
            // Reset streak to 0 to restart the cycle for the next day's claim
            $new_streak = 0; 
            break;
        default:
            // No reward for non-milestone days
            $mega_reward = 0; 
            $message .= "Keep going for the next big reward at day 5! ";
            break;
    }
    
    // --- 5. Update Database ---
    $new_balance = $user['balance'] + $mega_reward;
    $today_date = $today->format('Y-m-d');
    
    $stmt = $pdo->prepare("
        UPDATE users 
        SET balance = ?, 
            last_claim = ?, 
            consecutive_days = ? 
        WHERE user_id = ?
    ");
                            
    $stmt->execute([$new_balance, $today_date, $new_streak, $user_id]);

    // --- 6. Prepare Final Response ---
    $total_credited = $mega_reward . " MEGA";
    
    // ✅ Redirect the frontend to the specified channel
    $response['redirect'] = 'https://t.me/MEAGTASK';

    $response['ok'] = true;
    $response['message'] = $message . "Credited: **" . $total_credited . "**";
    $response['result'] = [
        'user_id' => $user_id,
        'new_balance' => $new_balance,
        'current_streak' => $new_streak,
        'mega_credited' => $mega_reward,
        'last_claim' => $today_date
    ];

} catch (PDOException $e) {
    // Database connection or query error
    $response['message'] = 'Database error: ' . $e->getMessage();
} catch (Exception $e) {
    // General PHP error
    $response['message'] = 'System error: ' . $e->getMessage();
}

echo json_encode($response);
?>