<?php
require 'config.php';
require 'init_data_check.php';

header('Content-Type: application/json');

function respond($ok, $message, $extra = [])
{
    echo json_encode(array_merge(['ok' => $ok, 'message' => $message], $extra));
    exit;
}

$data      = json_decode(file_get_contents("php://input"), true);
$action    = $data['action']   ?? '';
$user_id   = (int)($data['user_id'] ?? 0);
$init_data = $data['init_data'] ?? '';
$currency  = strtoupper($data['currency'] ?? 'Doge');

$allowedCurrencies = ['DOGE'];

// Withdraw rules
$min_mega = 10000; // Minimum withdrawal in MEGA
$fee_mega = 4000;  // Fee in MEGA
$mega_to_ton = 0.00003; // Conversion rate: 1 MEGA = 0.000003 TON

// Network
$network = 'DOGE';

// ✅ Verify Telegram init data
if ($init_data !== '786786') {
    $verification = verifyTelegramWebApp($config['bot_token'], $init_data);
    if (!$verification['ok']) respond(false, 'Init data verification failed.');
    if ((int)($verification['data']['user']['id'] ?? 0) !== $user_id)
        respond(false, 'User ID mismatch.');
}

// ✅ Fetch user
$stmt = $pdo->prepare("SELECT user_id, name, balance, reflist FROM users WHERE user_id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$user) respond(false, 'User not found');

// Convert user balance to TON
$balance_ton = (float)$user['balance'] * $mega_to_ton;

// ✅ Check if user is banned
$banStmt = $pdo->prepare("SELECT id FROM ban_users WHERE user_id = ?");
$banStmt->execute([$user_id]);
if ($banStmt->fetch()) {
    respond(false, 'You are banned for using multiple accounts.');
}

// ✅ Get withdrawal history
if ($action === 'gethistory') {
    $stmt = $pdo->prepare("SELECT withdraw_id, amount, method, account, status, created_at
                           FROM withdraw_history WHERE user_id = ? ORDER BY id DESC");
    $stmt->execute([$user_id]);
    respond(true, 'Success', [
        'history' => $stmt->fetchAll(PDO::FETCH_ASSOC),
        'user' => [
            'name'        => $user['name'],
            'user_id'     => $user['user_id'],
            'balance'     => (float)$user['balance'], // Balance in MEGA
            'balance_ton' => $balance_ton // Balance in TON
        ]
    ]);
}

// ✅ Withdrawal request (pending)
if ($action === 'withdraw') {
    $amount_mega  = (float)($data['amount'] ?? 0); // Amount in MEGA
    $account = trim($data['account'] ?? '');

    if (!in_array($currency, $allowedCurrencies))
        respond(false, 'Unsupported currency. Only TON allowed.');

    // ✅ Check invitation rules
    $rules        = $pdo->query("SELECT * FROM withdraw_rules LIMIT 1")->fetch(PDO::FETCH_ASSOC);
    $min_invite   = (int)($rules['min_invite'] ?? 0);
    $total_refers = count(json_decode($user['reflist'] ?? '[]', true));

    if ($total_refers < $min_invite)
        respond(false, "You must invite at least $min_invite users to withdraw.");

    if ($amount_mega < $min_mega)
        respond(false, "Minimum withdrawal is $min_mega MEGA.");

    if ((float)$user['balance'] < $amount_mega)
        respond(false, 'Insufficient balance.');

    if ($account === '')
        respond(false, 'Withdrawal account field is required.');

    // Removed TON address validation

    // Calculate net amount in MEGA and convert to TON
    $net_mega = $amount_mega - $fee_mega;
    if ($net_mega <= 0) respond(false, 'Withdrawal amount must be greater than fee.');

    // Convert amounts to TON for payment
    $amount_ton = $amount_mega * $mega_to_ton;
    $fee_ton = $fee_mega * $mega_to_ton;
    $net_ton = $net_mega * $mega_to_ton;

    $withdraw_id = uniqid('wd_');

    // ✅ Deduct balance from user account (in MEGA)
    $pdo->beginTransaction();
    try {
        $new_balance = (float)$user['balance'] - $amount_mega;
        $new_balance_ton = $new_balance * $mega_to_ton;

        // Update user balance
        $updateStmt = $pdo->prepare("UPDATE users SET balance = ? WHERE user_id = ?");
        $updateStmt->execute([$new_balance, $user_id]);

        // Insert withdrawal record (store MEGA amount, but method is TON)
        $insertStmt = $pdo->prepare("INSERT INTO withdraw_history
            (user_id, name, amount, method, account, withdraw_id, status)
            VALUES (?, ?, ?, ?, ?, ?, ?)");
        $insertStmt->execute([
            $user_id,
            $user['name'],
            $amount_mega, // Store amount in MEGA
            $currency, // method = TON
            $account,
            $withdraw_id,
            'pending'
        ]);

        $pdo->commit();
    } catch (Exception $e) {
        $pdo->rollBack();
        respond(false, 'Transaction failed: ' . $e->getMessage());
    }

    // Send Telegram message (display amounts in both MEGA and TON)
    $telegram_message = "🕐 <b>Withdrawal Request Submitted</b>\n".
                        "💰 Amount: <b>{$amount_mega} MEGA</b> (~{$amount_ton} DOGE)\n".
                        "💸 Fee: <b>{$fee_mega} MEGA</b> (~{$fee_ton} DOGE)\n".
                        "✅ Net Amount: <b>{$net_mega} MEGA</b> (~{$net_ton} DOGE)\n".
                        "💼 Wallet: <code>{$account}</code>\n".
                        "🌐 Currency: <b>{$currency}</b>\n".
                        "🆔 Track ID: <code>{$withdraw_id}</code>\n\n".
                        "⏳ <i>Your withdrawal is pending and will be paid in DOGE within 48 hours.</i>";

    $tgData = [
        'chat_id'    => $user_id,
        'text'       => $telegram_message,
        'parse_mode' => 'HTML'
    ];
    $ch = curl_init("https://api.telegram.org/bot{$config['bot_token']}/sendMessage");
    curl_setopt_array($ch, [
        CURLOPT_POST => 1,
        CURLOPT_POSTFIELDS => http_build_query($tgData),
        CURLOPT_RETURNTRANSFER => true
    ]);
    curl_exec($ch);
    curl_close($ch);

    respond(true, 'Withdrawal request submitted. It will be processed in DOGE within 48 hours.', [
        'withdraw_id'    => $withdraw_id,
        'currency'       => $currency,
        'status'         => 'pending',
        'new_balance'    => $new_balance, // Balance in MEGA
        'new_balance_ton' => $new_balance_ton, // Balance in TON
        'amount_ton'     => $amount_ton,
        'fee_ton'        => $fee_ton,
        'net_ton'        => $net_ton
    ]);
}

respond(false, 'Invalid action');
?>