<?php
// Disable error display in production, log instead
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

ob_start();

// Headers
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, DELETE');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

require_once 'config.php';

// Response function
function sendResponse($status, $message, $data = null, $httpCode = 200) {
    http_response_code($httpCode);
    $response = [
        'status' => $status,
        'message' => $message
    ];
    if ($data !== null) {
        $response['data'] = $data;
    }
    echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_THROW_ON_ERROR);
    ob_end_flush();
    exit;
}

// Validate admin pin
function validateAdminPin($pin) {
    global $config;
    return is_string($pin) && $pin === $config['admin_pin'];
}

// Check PDO
if (!isset($pdo) || !($pdo instanceof PDO)) {
    error_log("Database connection failed: PDO not initialized", 3, 'error.log');
    sendResponse('error', 'Database connection failed', null, 500);
}

// PDO UTF-8 settings
try {
    $pdo->exec("SET NAMES utf8mb4");
    $pdo->exec("SET CHARACTER SET utf8mb4");
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Failed to configure PDO for UTF-8: " . $e->getMessage(), 3, 'error.log');
    sendResponse('error', 'Database encoding error: ' . $e->getMessage(), null, 500);
}

$method = $_SERVER['REQUEST_METHOD'];
$path = isset($_SERVER['PATH_INFO']) ? trim($_SERVER['PATH_INFO'], '/') : '';

try {
    // Admin pin validation (for DELETE, optional for GET)
    $pin = isset($_GET['admin_pin']) ? trim($_GET['admin_pin']) : '';
    if ($method !== 'GET' && !validateAdminPin($pin)) {
        sendResponse('error', 'Invalid admin pin', null, 401);
    }

    switch ($method) {
        case 'GET':
            if ($path === 'users') {
                // Fetch all users
                $stmt = $pdo->query("SELECT id, name, user_id, ip_address, balance, total_deposit, scratch_card, created_at, reflist, completed_task_ids FROM users");
                $users = $stmt->fetchAll();

                foreach ($users as &$user) {
                    $user['name'] = mb_convert_encoding($user['name'], 'UTF-8', 'UTF-8');
                    $user['scratch_card'] = json_decode($user['scratch_card'], true) ?? [];
                    $user['reflist'] = json_decode($user['reflist'], true) ?? [];
                    $user['completed_task_ids'] = json_decode($user['completed_task_ids'], true) ?? [];

                    // ✅ Calculate total deposits of referred users
                    $total_depo_by_refers = 0.00;
                    if (!empty($user['reflist'])) {
                        $ref_user_ids = array_column($user['reflist'], 'user_id');
                        if (!empty($ref_user_ids)) {
                            $placeholders = implode(',', array_fill(0, count($ref_user_ids), '?'));
                            $stmt_deposit = $pdo->prepare("SELECT SUM(total_deposit) AS total FROM users WHERE user_id IN ($placeholders)");
                            $stmt_deposit->execute($ref_user_ids);
                            $result = $stmt_deposit->fetch();
                            $total_depo_by_refers = (float)($result['total'] ?? 0.00);
                        }
                    }
                    $user['total_depo_by_refers'] = number_format($total_depo_by_refers, 2, '.', '');
                }
                unset($user);

                sendResponse('success', 'Users fetched successfully', $users);
            } elseif (preg_match('/^users\/(\d+)$/', $path, $matches)) {
                // Fetch single user by user_id
                $user_id = filter_var($matches[1], FILTER_VALIDATE_INT);
                if ($user_id === false) {
                    sendResponse('error', 'Invalid user ID', null, 400);
                }

                $stmt = $pdo->prepare("SELECT id, name, user_id, ip_address, balance, total_deposit, scratch_card, created_at, reflist, completed_task_ids FROM users WHERE user_id = ?");
                $stmt->execute([$user_id]);
                $user = $stmt->fetch();

                if ($user) {
                    $user['name'] = mb_convert_encoding($user['name'], 'UTF-8', 'UTF-8');
                    $user['scratch_card'] = json_decode($user['scratch_card'], true) ?? [];
                    $user['reflist'] = json_decode($user['reflist'], true) ?? [];
                    $user['completed_task_ids'] = json_decode($user['completed_task_ids'], true) ?? [];

                    // ✅ Calculate total deposits of referred users
                    $total_depo_by_refers = 0.00;
                    if (!empty($user['reflist'])) {
                        $ref_user_ids = array_column($user['reflist'], 'user_id');
                        if (!empty($ref_user_ids)) {
                            $placeholders = implode(',', array_fill(0, count($ref_user_ids), '?'));
                            $stmt_deposit = $pdo->prepare("SELECT SUM(total_deposit) AS total FROM users WHERE user_id IN ($placeholders)");
                            $stmt_deposit->execute($ref_user_ids);
                            $result = $stmt_deposit->fetch();
                            $total_depo_by_refers = (float)($result['total'] ?? 0.00);
                        }
                    }
                    $user['total_depo_by_refers'] = number_format($total_depo_by_refers, 2, '.', '');

                    sendResponse('success', 'User fetched successfully', $user);
                } else {
                    sendResponse('error', 'User not found', null, 404);
                }
            } else {
                sendResponse('error', 'Invalid endpoint', null, 404);
            }
            break;

        case 'DELETE':
            if (preg_match('/^users\/(\d+)$/', $path, $matches)) {
                $user_id = filter_var($matches[1], FILTER_VALIDATE_INT);
                if ($user_id === false) {
                    sendResponse('error', 'Invalid user ID', null, 400);
                }

                $stmt = $pdo->prepare("SELECT id FROM users WHERE user_id = ?");
                $stmt->execute([$user_id]);
                $user = $stmt->fetch();

                if (!$user) {
                    sendResponse('error', 'User not found', null, 404);
                }

                $pdo->beginTransaction();
                try {
                    $stmt = $pdo->prepare("DELETE FROM withdraw_history WHERE user_id = ?");
                    $stmt->execute([$user_id]);

                    $stmt = $pdo->prepare("DELETE FROM users WHERE user_id = ?");
                    $stmt->execute([$user_id]);

                    $pdo->commit();
                    sendResponse('success', 'User deleted successfully');
                } catch (PDOException $e) {
                    $pdo->rollBack();
                    error_log("Delete failed: " . $e->getMessage(), 3, 'error.log');
                    sendResponse('error', 'Failed to delete user: ' . $e->getMessage(), null, 500);
                }
            } else {
                sendResponse('error', 'Invalid endpoint', null, 404);
            }
            break;

        default:
            sendResponse('error', 'Method not allowed', null, 405);
    }
} catch (Exception $e) {
    error_log("Server error: " . $e->getMessage(), 3, 'error.log');
    sendResponse('error', 'Server error: ' . $e->getMessage(), null, 500);
}

ob_end_flush();
?>
