<?php
require_once 'config.php';
header('Content-Type: application/json');

// ---- Read Request (JSON, POST, or GET) ----
$raw = file_get_contents('php://input');
$input = json_decode($raw, true);

if (!$input) { 
    $input = $_POST; 
}

$action = $input['action'] ?? ($_GET['action'] ?? '');
$admin_password = $input['admin_password'] ?? '';

// ---- Helper: Validate Admin ----
function validateAdminPassword($password) {
    global $config;
    return $password === $config['admin_pin'];
}

// ---- Helper: Generate Code ----
function generateRandomCode($length = 8) {
    $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $code = '';
    for ($i = 0; $i < $length; $i++) {
        $code .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $code;
}

// ---- Helper: Send to Telegram (using cURL) ----
function sendToTelegram($message) {
    global $config;

    $chat_id = '-100272939';
    $url = "https://api.telegram.org/bot" . $config['bot_token'] . "/sendMessage";

    $data = [
        'chat_id' => $chat_id,
        'text' => $message,
        'parse_mode' => 'HTML'
    ];

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);

    $response = curl_exec($ch);
    curl_close($ch);

    $json = json_decode($response, true);
    return $json['ok'] ?? false;
}

// ---- Default Response ----
$response = ['ok' => false, 'message' => '', 'result' => []];

try {

    switch ($action) {

        /*--------------------------------
         GET ALL CODES
        --------------------------------*/
        case 'get':
            if (!validateAdminPassword($admin_password)) {
                $response['message'] = 'Invalid admin password';
                break;
            }

            $stmt = $pdo->query("SELECT * FROM gift_codes ORDER BY id DESC");
            $response['ok'] = true;
            $response['result'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $response['message'] = 'Gift codes retrieved successfully';
            break;


        /*--------------------------------
         ADMIN CREATE CODE
        --------------------------------*/
        case 'create':
            if (!validateAdminPassword($admin_password)) {
                $response['message'] = 'Invalid admin password';
                break;
            }

            $gift_code = $input['gift_code'] ?? generateRandomCode();
            $prize = $input['prize'] ?? 0;
            $users = $input['users'] ?? 100;

            if ($prize <= 0 || $users <= 0) {
                $response['message'] = 'Invalid input parameters';
                break;
            }

            // Check if code exists
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM gift_codes WHERE gift_code = ?");
            $stmt->execute([$gift_code]);
            if ($stmt->fetchColumn() > 0) {
                $response['message'] = 'Gift code already exists';
                break;
            }

            $stmt = $pdo->prepare("INSERT INTO gift_codes (gift_code, prize, users) VALUES (?, ?, ?)");
            $stmt->execute([$gift_code, $prize, $users]);

            $message = "New Gift Code: <b>$gift_code</b>\nPrize: <b>$prize MEGA</b>\nMax Users: <b>$users</b>";
            sendToTelegram($message);

            $response['ok'] = true;
            $response['message'] = 'Gift code created';
            $response['result'] = [
                'gift_code' => $gift_code,
                'prize' => $prize,
                'users' => $users
            ];
            break;


        /*--------------------------------
         DELETE CODE
        --------------------------------*/
        case 'delete':
            if (!validateAdminPassword($admin_password)) {
                $response['message'] = 'Invalid admin password';
                break;
            }

            $gift_code = $input['gift_code'] ?? '';

            $stmt = $pdo->prepare("DELETE FROM gift_codes WHERE gift_code = ?");
            $stmt->execute([$gift_code]);

            $response['ok'] = $stmt->rowCount() > 0;
            $response['message'] = $response['ok'] ? 'Gift code deleted' : 'Gift code not found';
            break;


        /*--------------------------------
         UPDATE CODE
        --------------------------------*/
        case 'edit':
            if (!validateAdminPassword($admin_password)) {
                $response['message'] = 'Invalid admin password';
                break;
            }

            $gift_code = $input['gift_code'] ?? '';
            $prize = $input['prize'] ?? null;
            $users = $input['users'] ?? null;

            $updates = [];
            $params = [];

            if ($prize !== null) {
                $updates[] = "prize = ?";
                $params[] = $prize;
            }
            if ($users !== null) {
                $updates[] = "users = ?";
                $params[] = $users;
            }

            if (empty($updates)) {
                $response['message'] = 'No updates provided';
                break;
            }

            $params[] = $gift_code;

            $stmt = $pdo->prepare("UPDATE gift_codes SET ".implode(', ', $updates)." WHERE gift_code = ?");
            $stmt->execute($params);

            $response['ok'] = $stmt->rowCount() > 0;
            $response['message'] = $response['ok'] ? 'Gift code updated' : 'Gift code not found';
            break;


        /*--------------------------------
         PUBLIC USER GENERATE RANDOM CODE
        --------------------------------*/
        case 'public_generate':

            $prize = rand(50, 300);
            $users = 100;

            // Unique code loop
            do {
                $gift_code = generateRandomCode();
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM gift_codes WHERE gift_code = ?");
                $stmt->execute([$gift_code]);
                $exists = $stmt->fetchColumn() > 0;
            } while ($exists);

            // Insert into DB
            $stmt = $pdo->prepare("INSERT INTO gift_codes (gift_code, prize, users) VALUES (?, ?, ?)");
            $stmt->execute([$gift_code, $prize, $users]);

            $message = "New Gift Code: <b>$gift_code</b>\nPrize: <b>$prize MEGA</b>\nMax Users: <b>$users</b>";
            sendToTelegram($message);

            $response['ok'] = true;
            $response['message'] = "Gift code generated";
            $response['result'] = [
                'gift_code' => $gift_code,
                'prize' => $prize,
                'users' => $users
            ];
            break;

        /*--------------------------------
         REDEEM CODE
        --------------------------------*/
        case 'redeem':
            $gift_code = $input['gift_code'] ?? '';

            if (empty($gift_code)) {
                $response['message'] = 'Invalid input parameters';
                break;
            }

            $stmt = $pdo->prepare("SELECT * FROM gift_codes WHERE gift_code = ? AND users > 0");
            $stmt->execute([$gift_code]);
            $row = $stmt->fetch();

            if ($row) {
                $stmt = $pdo->prepare("UPDATE gift_codes SET users = users - 1 WHERE gift_code = ?");
                $stmt->execute([$gift_code]);

                $response['ok'] = true;
                $response['message'] = 'Code redeemed';
                $response['result'] = ['prize' => $row['prize']];
            } else {
                $response['message'] = 'Invalid or expired code';
            }
            break;


        /*--------------------------------
         INVALID
        --------------------------------*/
        default:
            $response['message'] = 'Invalid action';
    }

} catch (PDOException $e) {
    $response['message'] = "Database error: " . $e->getMessage();
}

echo json_encode($response);
?>